import { useState } from "react";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription } from "./ui/dialog";
import { Button } from "./ui/button";
import { Card, CardContent, CardHeader } from "./ui/card";
import { Badge } from "./ui/badge";
import { Avatar, AvatarFallback, AvatarImage } from "./ui/avatar";
import { Pagination, PaginationContent, PaginationItem, PaginationLink, PaginationNext, PaginationPrevious } from "./ui/pagination";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "./ui/select";
import { Star, Clock, MessageCircle, Mail, Phone, Filter } from "lucide-react";
import { toast } from "sonner@2.0.3";

interface Mentor {
  id: string;
  name: string;
  expertise: string[];
  rating: number;
  responseTime: string;
  hourlyRate: number;
  avatar?: string;
  isOnline: boolean;
  totalSessions: number;
  email: string;
  phone: string;
}

interface SupportModalProps {
  isOpen: boolean;
  onClose: () => void;
  onSelectMentor?: (mentor: { id: string; name: string; hourlyRate: number; expertise: string[]; }) => void;
  classType?: "crib" | "clicker" | "quest";
}

export function SupportModal({ isOpen, onClose, onSelectMentor, classType = "quest" }: SupportModalProps) {
  const [selectedMentor, setSelectedMentor] = useState<string | null>(null);
  const [currentPage, setCurrentPage] = useState(1);
  const [filterByExpertise, setFilterByExpertise] = useState<string>("all");
  const [sortBy, setSortBy] = useState<string>("rating");
  
  const mentorsPerPage = 3;

  // Get theme colors based on class type - matching parent profile color scheme
  const getThemeColors = () => {
    switch (classType) {
      case "crib":
        return {
          primary: "text-crib-orange",
          bg: "bg-crib-orange",
          hover: "hover:bg-crib-orange/90",
          headerGradient: "from-yellow via-light-blue to-navy",
          cardGradient: "from-light-blue to-yellow",
          accent: "bg-crib-star-yellow",
          icon: "text-crib-teal",
          border: "border-crib-orange/30",
          textColor: "text-navy"
        };
      case "clicker":
        return {
          primary: "text-light-blue",
          bg: "bg-light-blue",
          hover: "hover:bg-light-blue/90",
          headerGradient: "from-light-blue to-navy",
          cardGradient: "from-navy to-light-blue",
          accent: "bg-yellow",
          icon: "text-yellow",
          border: "border-light-blue/30",
          textColor: "text-white"
        };
      case "quest":
        return {
          primary: "text-light-blue",
          bg: "bg-navy",
          hover: "hover:bg-navy/90",
          headerGradient: "from-navy via-navy/90 to-navy/80",
          cardGradient: "from-navy to-light-blue",
          accent: "bg-light-blue",
          icon: "text-light-blue",
          border: "border-navy/30",
          textColor: "text-white"
        };
    }
  };

  const theme = getThemeColors();

  const allMentors: Mentor[] = [
    {
      id: "1",
      name: "Sarah Johnson",
      expertise: ["Pictoblox", "Python", "Scratch"],
      rating: 4.9,
      responseTime: "< 5 min",
      hourlyRate: 25,
      isOnline: true,
      totalSessions: 127,
      email: "sarah.johnson@digifunzi.com",
      phone: "+1 (555) 123-4567",
      avatar: "https://images.unsplash.com/photo-1494790108755-2616b612b47c?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&w=150"
    },
    {
      id: "2", 
      name: "Michael Chen",
      expertise: ["AI/ML", "Python", "Web Development"],
      rating: 4.8,
      responseTime: "< 10 min",
      hourlyRate: 30,
      isOnline: true,
      totalSessions: 89,
      email: "michael.chen@digifunzi.com",
      phone: "+1 (555) 234-5678",
      avatar: "https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&w=150"
    },
    {
      id: "3",
      name: "Emma Rodriguez",
      expertise: ["Game Development", "Pictoblox", "Creative Coding"],
      rating: 4.7,
      responseTime: "< 15 min", 
      hourlyRate: 22,
      isOnline: false,
      totalSessions: 156,
      email: "emma.rodriguez@digifunzi.com",
      phone: "+1 (555) 345-6789",
      avatar: "https://images.unsplash.com/photo-1438761681033-6461ffad8d80?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&w=150"
    },
    {
      id: "4",
      name: "David Kim",
      expertise: ["Hardware Programming", "Robotics", "IoT"],
      rating: 4.9,
      responseTime: "< 8 min",
      hourlyRate: 35,
      isOnline: true,
      totalSessions: 203,
      email: "david.kim@digifunzi.com",
      phone: "+1 (555) 456-7890",
      avatar: "https://images.unsplash.com/photo-1472099645785-5658abf4ff4e?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&w=150"
    },
    {
      id: "5",
      name: "Lisa Anderson",
      expertise: ["Scratch", "Python", "Creative Coding"],
      rating: 4.6,
      responseTime: "< 12 min",
      hourlyRate: 28,
      isOnline: true,
      totalSessions: 98,
      email: "lisa.anderson@digifunzi.com",
      phone: "+1 (555) 567-8901",
      avatar: "https://images.unsplash.com/photo-1517841905240-472988babdf9?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&w=150"
    },
    {
      id: "6",
      name: "James Wilson",
      expertise: ["Web Development", "JavaScript", "React"],
      rating: 4.8,
      responseTime: "< 7 min",
      hourlyRate: 32,
      isOnline: false,
      totalSessions: 145,
      email: "james.wilson@digifunzi.com",
      phone: "+1 (555) 678-9012",
      avatar: "https://images.unsplash.com/photo-1500648767791-00dcc994a43e?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&w=150"
    },
    {
      id: "7",
      name: "Maria Garcia",
      expertise: ["AI/ML", "Robotics", "Python"],
      rating: 4.9,
      responseTime: "< 6 min",
      hourlyRate: 38,
      isOnline: true,
      totalSessions: 178,
      email: "maria.garcia@digifunzi.com",
      phone: "+1 (555) 789-0123",
      avatar: "https://images.unsplash.com/photo-1544005313-94ddf0286df2?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&w=150"
    },
    {
      id: "8",
      name: "Robert Taylor",
      expertise: ["Game Development", "Unity", "C#"],
      rating: 4.7,
      responseTime: "< 11 min",
      hourlyRate: 29,
      isOnline: true,
      totalSessions: 112,
      email: "robert.taylor@digifunzi.com",
      phone: "+1 (555) 890-1234",
      avatar: "https://images.unsplash.com/photo-1566492031773-4f4e44671d66?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&w=150"
    }
  ];

  // Get unique expertise areas for filtering
  const uniqueExpertise = Array.from(
    new Set(allMentors.flatMap(mentor => mentor.expertise))
  );

  // Filter and sort mentors
  const getFilteredAndSortedMentors = () => {
    let filtered = allMentors;

    // Filter by expertise
    if (filterByExpertise !== "all") {
      filtered = filtered.filter(mentor => 
        mentor.expertise.includes(filterByExpertise)
      );
    }

    // Sort mentors
    filtered.sort((a, b) => {
      switch (sortBy) {
        case "rating":
          return b.rating - a.rating;
        case "hourlyRate":
          return a.hourlyRate - b.hourlyRate;
        case "experience":
          return b.totalSessions - a.totalSessions;
        case "responseTime":
          return parseInt(a.responseTime) - parseInt(b.responseTime);
        default:
          return 0;
      }
    });

    return filtered;
  };

  const filteredMentors = getFilteredAndSortedMentors();
  const totalPages = Math.ceil(filteredMentors.length / mentorsPerPage);
  const startIndex = (currentPage - 1) * mentorsPerPage;
  const currentMentors = filteredMentors.slice(startIndex, startIndex + mentorsPerPage);

  const handleSelectMentor = (mentorId: string, mentorName: string) => {
    setSelectedMentor(mentorId);
    
    if (onSelectMentor) {
      const mentor = allMentors.find(m => m.id === mentorId);
      if (mentor) {
        onSelectMentor({
          id: mentor.id,
          name: mentor.name,
          hourlyRate: mentor.hourlyRate,
          expertise: mentor.expertise
        });
        onClose();
        return;
      }
    }
    
    toast.success(`Connected with ${mentorName}! They will reach out shortly.`);
    onClose();
  };

  const handleContactMethod = (method: string, mentorName: string, contact: string) => {
    if (method === "email") {
      window.open(`mailto:${contact}`, '_blank');
    } else if (method === "phone") {
      window.open(`tel:${contact}`, '_blank');
    }
    toast.success(`Opening ${method} for ${mentorName}...`);
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-7xl
                               max-h-[90vh]
                               overflow-y-auto
                               w-[95vw]">
        <DialogHeader className={`-mx-6 -mt-6 px-6 pt-6 pb-4 bg-gradient-to-r ${theme.headerGradient} ${theme.textColor} rounded-t-lg`}>
          <DialogTitle className="flex
                                 items-center
                                 gap-2
                                 text-2xl">
            <MessageCircle size={28} />
            {classType === "crib" ? "🌟 Get Magic Helper" :
             classType === "clicker" ? "🚀 Get Tech Support" :
             "Get Technical Support"}
          </DialogTitle>
          <DialogDescription className={classType === "crib" ? "text-navy/90" : "text-white/90"}>
            {classType === "crib" ? "Connect with our friendly magic helpers who will make learning super fun!" :
             classType === "clicker" ? "Connect with our expert tech mentors and level up your coding skills!" :
             "Connect with our expert mentors for personalized help with your coding journey"}
          </DialogDescription>
        </DialogHeader>

        {/* Filters and Controls */}
        <div className="flex
                       flex-col
                       md:flex-row
                       gap-4
                       mb-6
                       p-4
                       bg-gray-50
                       rounded-lg">
          <div className="flex
                         items-center
                         gap-2
                         flex-1">
            <Filter size={16} className="text-gray-600" />
            <Select value={filterByExpertise} onValueChange={setFilterByExpertise}>
              <SelectTrigger className="w-48">
                <SelectValue placeholder="Filter by expertise" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Expertise</SelectItem>
                {uniqueExpertise.map(skill => (
                  <SelectItem key={skill} value={skill}>{skill}</SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
          
          <div className="flex
                         items-center
                         gap-2
                         flex-1">
            <span className="text-sm
                           text-gray-600">Sort by:</span>
            <Select value={sortBy} onValueChange={setSortBy}>
              <SelectTrigger className="w-48">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="rating">Highest Rating</SelectItem>
                <SelectItem value="hourlyRate">Lowest Price</SelectItem>
                <SelectItem value="experience">Most Experience</SelectItem>
                <SelectItem value="responseTime">Fastest Response</SelectItem>
              </SelectContent>
            </Select>
          </div>
          
          <div className="text-sm
                         text-gray-600
                         flex
                         items-center">
            Showing {currentMentors.length} of {filteredMentors.length} mentors
          </div>
        </div>

        {/* Mentors Grid */}
        <div className="space-y-6
                       mb-6">
          {currentMentors.map((mentor) => (
            <Card key={mentor.id} className="relative
                                           hover:shadow-lg
                                           transition-shadow
                                           p-4">
              {mentor.isOnline && (
                <div className="absolute
                               top-3
                               right-3
                               z-10">
                  <Badge variant="secondary" className="bg-green-100
                                                       text-green-700
                                                       border-green-200
                                                       text-xs
                                                       px-2
                                                       py-1">
                    • Online
                  </Badge>
                </div>
              )}
              
              {/* Header Section */}
              <div className="flex
                             items-start
                             gap-3
                             mb-3">
                <Avatar className="h-14
                                 w-14
                                 border-2
                                 border-gray-100
                                 flex-shrink-0">
                  <AvatarImage src={mentor.avatar} />
                  <AvatarFallback className={`bg-gradient-to-br ${theme.cardGradient}
                                           ${theme.textColor}`}>
                    {mentor.name.split(' ').map(n => n[0]).join('')}
                  </AvatarFallback>
                </Avatar>
                
                <div className="flex-1
                               min-w-0
                               pr-8">
                  <h3 className="font-semibold
                                mb-1
                                truncate">{mentor.name}</h3>
                  
                  <div className="flex
                                 items-center
                                 gap-2
                                 mb-2
                                 text-sm">
                    <div className="flex
                                   items-center
                                   gap-1">
                      <Star className="text-yellow-400
                                     fill-current" size={14} />
                      <span className="font-medium">{mentor.rating}</span>
                    </div>
                    <span className="text-gray-400">•</span>
                    <span className="text-gray-600">{mentor.totalSessions} sessions</span>
                  </div>
                  
                  <div className="flex
                                 items-center
                                 justify-between
                                 text-sm">
                    <div className="flex
                                   items-center
                                   gap-1
                                   text-gray-600">
                      <Clock size={14} />
                      <span>{mentor.responseTime}</span>
                    </div>
                    <span className="font-semibold
                                   text-navy">${mentor.hourlyRate}/hr</span>
                  </div>
                </div>
              </div>
              
              {/* Expertise Tags */}
              <div className="flex
                             flex-wrap
                             gap-1.5
                             mb-3">
                {mentor.expertise.slice(0, 3).map((skill) => (
                  <Badge key={skill} variant="outline" className="text-xs
                                                                 px-2
                                                                 py-1
                                                                 h-6">
                    {skill}
                  </Badge>
                ))}
                {mentor.expertise.length > 3 && (
                  <Badge variant="outline" className="text-xs
                                                    px-2
                                                    py-1
                                                    h-6
                                                    text-gray-500">
                    +{mentor.expertise.length - 3}
                  </Badge>
                )}
              </div>
              
              {/* Contact Information */}
              <div className="space-y-1.5
                             mb-4
                             p-2.5
                             bg-gray-50
                             rounded-md">
                <div className="flex
                               items-center
                               gap-2
                               text-xs">
                  <Mail size={12} className="text-gray-500
                                           flex-shrink-0" />
                  <span className="text-gray-700
                                 truncate">{mentor.email}</span>
                </div>
                <div className="flex
                               items-center
                               gap-2
                               text-xs">
                  <Phone size={12} className="text-gray-500
                                            flex-shrink-0" />
                  <span className="text-gray-700">{mentor.phone}</span>
                </div>
              </div>
              
              {/* Action Buttons */}
              <div className="space-y-2">
                <div className="grid
                               grid-cols-2
                               gap-2">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handleContactMethod("email", mentor.name, mentor.email)}
                    className="flex
                             items-center
                             gap-1.5
                             h-8
                             text-xs"
                  >
                    <Mail size={12} />
                    Email
                  </Button>
                  
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handleContactMethod("phone", mentor.name, mentor.phone)}
                    className="flex
                             items-center
                             gap-1.5
                             h-8
                             text-xs"
                  >
                    <Phone size={12} />
                    Call
                  </Button>
                </div>
                
                <Button
                  className={`w-full
                           ${theme.bg}
                           ${theme.hover}
                           text-white
                           h-9
                           text-sm`}
                  onClick={() => handleSelectMentor(mentor.id, mentor.name)}
                  disabled={!mentor.isOnline}
                >
                  {onSelectMentor ? 
                    (classType === "crib" ? "🌟 Book Helper" : classType === "clicker" ? "⚡ Book Session" : "Book Session") : 
                    mentor.isOnline ? 
                      (classType === "crib" ? "Pick Helper" : classType === "clicker" ? "Choose Mentor" : "Select Mentor") : 
                      "Currently Offline"}
                </Button>
              </div>
            </Card>
          ))}
        </div>
        
        {/* Pagination */}
        {totalPages > 1 && (
          <div className="flex
                         justify-center
                         mt-6">
            <Pagination>
              <PaginationContent>
                <PaginationItem>
                  <PaginationPrevious 
                    onClick={() => setCurrentPage(Math.max(1, currentPage - 1))}
                    className={currentPage === 1 ? "pointer-events-none opacity-50" : "cursor-pointer"}
                  />
                </PaginationItem>
                
                {Array.from({ length: totalPages }, (_, i) => i + 1).map((page) => (
                  <PaginationItem key={page}>
                    <PaginationLink
                      onClick={() => setCurrentPage(page)}
                      isActive={currentPage === page}
                      className="cursor-pointer"
                    >
                      {page}
                    </PaginationLink>
                  </PaginationItem>
                ))}
                
                <PaginationItem>
                  <PaginationNext 
                    onClick={() => setCurrentPage(Math.min(totalPages, currentPage + 1))}
                    className={currentPage === totalPages ? "pointer-events-none opacity-50" : "cursor-pointer"}
                  />
                </PaginationItem>
              </PaginationContent>
            </Pagination>
          </div>
        )}
        
        <div className="flex
                       justify-end
                       gap-2
                       mt-6
                       pt-4
                       border-t">
          <Button variant="outline" onClick={onClose}>
            Cancel
          </Button>
        </div>
      </DialogContent>
    </Dialog>
  );
}