import { useState } from "react";
import { Upload, FileText, Download, Eye, Clock, CheckCircle, AlertCircle, Calendar, Award } from "lucide-react";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "./ui/card";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Textarea } from "./ui/textarea";
import { toast } from "sonner@2.0.3";

export function ProjectsView({ selectedClass }) {
  const [selectedAssignment, setSelectedAssignment] = useState(null);
  const [showUploadModal, setShowUploadModal] = useState(false);
  const [uploadFile, setUploadFile] = useState(null);
  const [uploadNotes, setUploadNotes] = useState("");

  // Sample assignments data - would come from API in real implementation
  const assignments = [
    {
      id: "assign-1",
      courseTitle: "UI/UX Design with Figma and Miro",
      sessionNumber: 13,
      assignmentTitle: "Design a Mobile App Interface",
      description: "Create a complete mobile app interface for a weather application using Figma",
      dueDate: "2025-05-10",
      submittedDate: "2025-05-02",
      status: "graded",
      grade: "Excellent",
      feedback: "Outstanding work! Your color choices and user flow are very impressive.",
      classType: "clicker",
      fileSubmitted: {
        name: "weather-app-design.fig",
        size: "2.4 MB",
        type: "Figma File"
      }
    },
    {
      id: "assign-2", 
      courseTitle: "UI/UX Design with Figma and Miro",
      sessionNumber: 12,
      assignmentTitle: "User Journey Mapping",
      description: "Create a detailed user journey map for an e-commerce website",
      dueDate: "2025-05-08",
      submittedDate: "2025-05-02",
      status: "graded",
      grade: "Good", 
      feedback: "Good understanding of user flows. Consider adding more detail to pain points.",
      classType: "clicker",
      fileSubmitted: {
        name: "user-journey-map.pdf",
        size: "1.8 MB",
        type: "PDF Document"
      }
    },
    {
      id: "assign-3",
      courseTitle: "UI/UX Design with Figma and Miro", 
      sessionNumber: 11,
      assignmentTitle: "Wireframe Creation",
      description: "Design low-fidelity wireframes for a restaurant booking app",
      dueDate: "2025-05-06",
      submittedDate: "2025-05-02",
      status: "graded",
      grade: "Good",
      feedback: "Clear wireframes with good layout structure. Try to include more annotations.",
      classType: "clicker",
      fileSubmitted: {
        name: "restaurant-wireframes.png",
        size: "956 KB", 
        type: "Image File"
      }
    },
    {
      id: "assign-4",
      courseTitle: "Game Builder Academy",
      sessionNumber: 8,
      assignmentTitle: "Create Your First Platform Game",
      description: "Build a simple platform game with at least 3 levels using PictoBlox",
      dueDate: "2025-05-12",
      status: "pending",
      classType: "clicker"
    },
    {
      id: "assign-5", 
      courseTitle: "Website Wizard Course",
      sessionNumber: 5,
      assignmentTitle: "Personal Portfolio Website",
      description: "Create a responsive personal portfolio website showcasing your projects",
      dueDate: "2025-05-15",
      status: "pending",
      classType: "clicker"
    }
  ];

  // Filter assignments by selected class
  const filteredAssignments = assignments.filter(assignment => assignment.classType === selectedClass);

  const getStatusColor = (status) => {
    switch (status) {
      case "graded":
        return "bg-green-100 text-green-800";
      case "submitted":
        return "bg-blue-100 text-blue-800";
      case "pending":
        return "bg-yellow-100 text-yellow-800";
      case "overdue":
        return "bg-red-100 text-red-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const getGradeColor = (grade) => {
    switch (grade.toLowerCase()) {
      case "excellent":
        return "bg-green-100 text-green-800";
      case "good":
        return "bg-blue-100 text-blue-800";
      case "fair":
        return "bg-yellow-100 text-yellow-800";
      case "needs improvement":
        return "bg-red-100 text-red-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const getClassTheme = () => {
    switch (selectedClass) {
      case "crib":
        return {
          gradient: "from-yellow via-light-blue to-navy",
          accent: "light-blue",
          name: "Crib Adventures"
        };
      case "clicker":
        return {
          gradient: "from-light-blue to-navy",
          accent: "yellow", 
          name: "Clicker Academy"
        };
      case "quest":
        return {
          gradient: "",
          accent: "light-blue",
          name: "Quest Mastery",
          solidBg: "bg-navy"
        };
    }
  };

  const theme = getClassTheme();

  const handleFileUpload = (event) => {
    const file = event.target.files?.[0];
    if (file) {
      setUploadFile(file);
    }
  };

  const handleSubmitAssignment = () => {
    if (!uploadFile || !selectedAssignment) {
      toast.error("Please select a file to upload");
      return;
    }

    // In real implementation, this would upload to server
    toast.success(`Assignment submitted successfully! File: ${uploadFile.name}`);
    setShowUploadModal(false);
    setUploadFile(null);
    setUploadNotes("");
    setSelectedAssignment(null);
  };

  const openUploadModal = (assignment) => {
    setSelectedAssignment(assignment);
    setShowUploadModal(true);
  };

  return (
    <div className="p-6">
      {/* Header */}
      <div className="mb-6">
        <div className={`p-6 rounded-lg text-white mb-6 ${theme.solidBg || `bg-gradient-to-r ${theme.gradient}`}`}>
          <h1 className="text-3xl mb-2">📋 My Projects & Assignments</h1>
          <p className="text-white/90">Track your coursework, submit assignments, and view feedback from {theme.name}</p>
        </div>
      </div>

      {/* Statistics Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-6">
        <Card>
          <CardContent className="p-4 text-center">
            <div className={`w-12 h-12 rounded-full bg-${theme.accent}/10 flex items-center justify-center mx-auto mb-2`}>
              <FileText className={`w-6 h-6 text-${theme.accent}`} />
            </div>
            <p className="text-2xl font-bold text-navy">{filteredAssignments.length}</p>
            <p className="text-sm text-gray-600">Total Assignments</p>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4 text-center">
            <div className="w-12 h-12 rounded-full bg-green-100 flex items-center justify-center mx-auto mb-2">
              <CheckCircle className="w-6 h-6 text-green-600" />
            </div>
            <p className="text-2xl font-bold text-navy">{filteredAssignments.filter(a => a.status === "graded").length}</p>
            <p className="text-sm text-gray-600">Completed</p>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4 text-center">
            <div className="w-12 h-12 rounded-full bg-yellow-100 flex items-center justify-center mx-auto mb-2">
              <Clock className="w-6 h-6 text-yellow-600" />
            </div>
            <p className="text-2xl font-bold text-navy">{filteredAssignments.filter(a => a.status === "pending").length}</p>
            <p className="text-sm text-gray-600">Pending</p>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4 text-center">
            <div className="w-12 h-12 rounded-full bg-blue-100 flex items-center justify-center mx-auto mb-2">
              <Award className="w-6 h-6 text-blue-600" />
            </div>
            <p className="text-2xl font-bold text-navy">{filteredAssignments.filter(a => a.grade === "Excellent").length}</p>
            <p className="text-sm text-gray-600">Excellent Grades</p>
          </CardContent>
        </Card>
      </div>

      {/* Assignments Table */}
      <Card>
        <CardHeader>
          <CardTitle>Assignment Submissions</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead>
                <tr className="border-b">
                  <th className="text-left p-3">Class</th>
                  <th className="text-left p-3">Course</th>
                  <th className="text-left p-3">Assignment</th>
                  <th className="text-left p-3">Status</th>
                  <th className="text-left p-3">Due Date</th>
                  <th className="text-left p-3">Grade</th>
                  <th className="text-left p-3">Action</th>
                </tr>
              </thead>
              <tbody>
                {filteredAssignments.map((assignment) => (
                  <tr key={assignment.id} className="border-b hover:bg-gray-50">
                    <td className="p-3">
                      <Badge className={`bg-${theme.accent}/10 text-${theme.accent}`}>
                        {selectedClass.charAt(0).toUpperCase() + selectedClass.slice(1)}
                      </Badge>
                    </td>
                    <td className="p-3">
                      <div>
                        <p className="font-medium text-navy">{assignment.courseTitle}</p>
                        <p className="text-sm text-gray-600">Session {assignment.sessionNumber}</p>
                      </div>
                    </td>
                    <td className="p-3">
                      <div>
                        <p className="font-medium">{assignment.assignmentTitle}</p>
                        <p className="text-sm text-gray-600 line-clamp-1">{assignment.description}</p>
                      </div>
                    </td>
                    <td className="p-3">
                      <Badge className={getStatusColor(assignment.status)}>
                        {assignment.status.charAt(0).toUpperCase() + assignment.status.slice(1)}
                      </Badge>
                    </td>
                    <td className="p-3">
                      <div className="flex items-center gap-2">
                        <Calendar className="w-4 h-4 text-gray-500" />
                        <span className="text-sm">{assignment.dueDate}</span>
                      </div>
                      {assignment.submittedDate && (
                        <p className="text-xs text-gray-500">Submitted: {assignment.submittedDate}</p>
                      )}
                    </td>
                    <td className="p-3">
                      {assignment.grade ? (
                        <Badge className={getGradeColor(assignment.grade)}>
                          {assignment.grade}
                        </Badge>
                      ) : (
                        <span className="text-gray-400">-</span>
                      )}
                    </td>
                    <td className="p-3">
                      <div className="flex gap-2">
                        {assignment.status === "pending" ? (
                          <Button
                            size="sm"
                            onClick={() => openUploadModal(assignment)}
                            className={`bg-${theme.accent} hover:bg-${theme.accent}/90 text-navy`}
                          >
                            <Upload className="w-4 h-4 mr-1" />
                            Submit
                          </Button>
                        ) : (
                          <>
                            {assignment.fileSubmitted && (
                              <Button size="sm" variant="outline">
                                <Download className="w-4 h-4 mr-1" />
                                Download
                              </Button>
                            )}
                            {assignment.feedback && (
                              <Button size="sm" variant="outline">
                                <Eye className="w-4 h-4 mr-1" />
                                View Feedback
                              </Button>
                            )}
                          </>
                        )}
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </CardContent>
      </Card>

      {/* Upload Modal */}
      {showUploadModal && selectedAssignment && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 w-full max-w-md mx-4">
            <h3 className="text-lg font-bold mb-4">Submit Assignment</h3>
            <p className="text-sm text-gray-600 mb-4">{selectedAssignment.assignmentTitle}</p>
            
            <div className="space-y-4">
              <div>
                <Label htmlFor="file-upload">Select File</Label>
                <Input
                  id="file-upload"
                  type="file"
                  onChange={handleFileUpload}
                  accept=".pdf,.doc,.docx,.png,.jpg,.jpeg,.fig,.sketch,.psd,.zip"
                />
                {uploadFile && (
                  <p className="text-sm text-green-600 mt-1">Selected: {uploadFile.name}</p>
                )}
              </div>
              
              <div>
                <Label htmlFor="notes">Additional Notes (Optional)</Label>
                <Textarea
                  id="notes"
                  placeholder="Any additional information about your submission..."
                  value={uploadNotes}
                  onChange={(e) => setUploadNotes(e.target.value)}
                />
              </div>
            </div>

            <div className="flex gap-3 mt-6">
              <Button
                onClick={handleSubmitAssignment}
                className={`flex-1 bg-${theme.accent} hover:bg-${theme.accent}/90 text-navy`}
              >
                Submit Assignment
              </Button>
              <Button
                variant="outline"
                onClick={() => {
                  setShowUploadModal(false);
                  setUploadFile(null);
                  setUploadNotes("");
                  setSelectedAssignment(null);
                }}
              >
                Cancel
              </Button>
            </div>
          </div>
        </div>
      )}

      {/* No assignments message */}
      {filteredAssignments.length === 0 && (
        <Card>
          <CardContent className="p-8 text-center">
            <FileText className="w-16 h-16 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">No assignments yet</h3>
            <p className="text-gray-600">Assignments will appear here as you progress through your courses.</p>
          </CardContent>
        </Card>
      )}
    </div>
  );
}
