import { useState, useEffect } from "react";
import { Card, CardContent, CardHeader } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Maximize2, Minimize2, ExternalLink, Play, Clock } from "lucide-react";
import { toast } from "sonner@2.0.3";

export function PictobloxEmbed({ className = "", sessionNumber, onPictobloxOpen }) {
  const [isMinimized, setIsMinimized] = useState(false);
  const [isFullscreen, setIsFullscreen] = useState(false);

  // Sample PictoBlox projects based on session
  const getProjectForSession = (session) => {
    const projects = {
      1: { name: "Getting Started", description: "Learn basic block programming" },
      2: { name: "Sprite Movement", description: "Make characters move around" },
      3: { name: "Interactive Story", description: "Create an interactive narrative" },
      4: { name: "Simple Game", description: "Build your first game" },
      5: { name: "Animation Project", description: "Create smooth animations" }
    };
    return projects[session] || { name: "Advanced Project", description: "Advanced programming concepts" };
  };

  const currentProject = getProjectForSession(sessionNumber);

  const handleToggleSize = () => {
    if (isFullscreen) {
      setIsFullscreen(false);
    } else if (isMinimized) {
      setIsMinimized(false);
    } else {
      setIsMinimized(true);
    }
  };

  const handleMaximize = () => {
    setIsFullscreen(true);
    setIsMinimized(false);
  };

  const openInNewTab = () => {
    // In a real implementation, this would open PictoBlox in a new tab
    window.open("https://pictoblox.ai/", "_blank");
  };

  if (isFullscreen) {
    return (
      <div className="fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
        <div className="bg-white rounded-lg w-full h-full max-w-7xl max-h-full flex flex-col">
          <div className="flex items-center justify-between p-4 border-b">
            <div>
              <h3 className="font-medium">PictoBlox - {currentProject.name}</h3>
              <p className="text-sm text-gray-600">{currentProject.description}</p>
            </div>
            <div className="flex items-center gap-2">
              <Button
                variant="outline"
                size="sm"
                onClick={openInNewTab}
              >
                <ExternalLink size={16} />
              </Button>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setIsFullscreen(false)}
              >
                <Minimize2 size={16} />
              </Button>
            </div>
          </div>
          <div className="flex-1 bg-gray-100 flex items-center justify-center">
            <div className="text-center">
              <div className="w-20 h-20 bg-light-blue rounded-full flex items-center justify-center mx-auto mb-4">
                <Play className="text-white" size={32} />
              </div>
              <h4 className="font-medium mb-2">PictoBlox Editor</h4>
              <p className="text-gray-600 mb-4">
                Interactive coding environment for Session {sessionNumber}
              </p>
              <Button onClick={openInNewTab} className="bg-light-blue hover:bg-light-blue/90">
                <ExternalLink size={16} className="mr-2" />
                Open in PictoBlox
              </Button>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <Card className={`${className} ${isMinimized ? "h-16" : "h-96"} transition-all duration-300`}>
      <CardHeader className="p-3">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-2">
            <Badge variant="secondary" className="bg-light-blue/10 text-light-blue">
              PictoBlox
            </Badge>
            {!isMinimized && (
              <div>
                <h4 className="font-medium text-sm">{currentProject.name}</h4>
                <p className="text-xs text-gray-600">{currentProject.description}</p>
              </div>
            )}
          </div>
          <div className="flex items-center gap-1">
            <Button
              variant="ghost"
              size="sm"
              onClick={openInNewTab}
              className="h-8 w-8 p-0"
            >
              <ExternalLink size={14} />
            </Button>
            <Button
              variant="ghost"
              size="sm"
              onClick={handleMaximize}
              className="h-8 w-8 p-0"
            >
              <Maximize2 size={14} />
            </Button>
            <Button
              variant="ghost"
              size="sm"
              onClick={handleToggleSize}
              className="h-8 w-8 p-0"
            >
              {isMinimized ? <Maximize2 size={14} /> : <Minimize2 size={14} />}
            </Button>
          </div>
        </div>
      </CardHeader>
      
      {!isMinimized && (
        <CardContent className="p-3 pt-0 h-full">
          <div className="h-full bg-gray-100 rounded-lg flex items-center justify-center">
            <div className="text-center">
              <div className="w-16 h-16 bg-light-blue rounded-full flex items-center justify-center mx-auto mb-3">
                <Play className="text-white" size={24} />
              </div>
              <h4 className="font-medium mb-2">PictoBlox Editor</h4>
              <p className="text-sm text-gray-600 mb-3">
                Interactive coding environment for Session {sessionNumber}
              </p>
              <Button 
                size="sm" 
                onClick={openInNewTab}
                className="bg-light-blue hover:bg-light-blue/90"
              >
                <ExternalLink size={14} className="mr-2" />
                Open Editor
              </Button>
            </div>
          </div>
        </CardContent>
      )}
    </Card>
  );
}