import { useState } from "react";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription } from "./ui/dialog";
import { Button } from "./ui/button";
import { Card, CardContent, CardHeader } from "./ui/card";
import { Badge } from "./ui/badge";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { CreditCard, Smartphone, DollarSign, Check, X } from "lucide-react";
import { toast } from "sonner@2.0.3";

interface PaymentModalProps {
  isOpen: boolean;
  onClose: () => void;
  courseTitle: string;
  price: number;
  onPaymentSuccess: () => void;
  sessionQuantity?: number;
  sessionDuration?: number;
  classType?: "crib" | "clicker" | "quest";
}

export function PaymentModal({ 
  isOpen, 
  onClose, 
  courseTitle, 
  price, 
  onPaymentSuccess,
  sessionQuantity,
  sessionDuration,
  classType = "quest"
}: PaymentModalProps) {
  const [paymentMethod, setPaymentMethod] = useState("mpesa");
  const [phoneNumber, setPhoneNumber] = useState("");
  const [cardNumber, setCardNumber] = useState("");
  const [expiryDate, setExpiryDate] = useState("");
  const [cvv, setCvv] = useState("");
  const [isProcessing, setIsProcessing] = useState(false);

  // Get theme colors based on class type - matching parent profile color scheme
  const getThemeColors = () => {
    switch (classType) {
      case "crib":
        return {
          primary: "text-crib-orange",
          bg: "bg-crib-orange",
          hover: "hover:bg-crib-orange/90",
          headerGradient: "from-yellow via-light-blue to-navy",
          cardGradient: "from-light-blue to-yellow",
          accent: "bg-crib-star-yellow",
          icon: "text-crib-teal",
          border: "border-crib-orange/30",
          textColor: "text-navy"
        };
      case "clicker":
        return {
          primary: "text-light-blue",
          bg: "bg-light-blue",
          hover: "hover:bg-light-blue/90",
          headerGradient: "from-light-blue to-navy",
          cardGradient: "from-navy to-light-blue",
          accent: "bg-yellow",
          icon: "text-yellow",
          border: "border-light-blue/30",
          textColor: "text-white"
        };
      case "quest":
        return {
          primary: "text-light-blue",
          bg: "bg-navy",
          hover: "hover:bg-navy/90",
          headerGradient: "from-navy via-navy/90 to-navy/80",
          cardGradient: "from-navy to-light-blue",
          accent: "bg-light-blue",
          icon: "text-light-blue",
          border: "border-navy/30",
          textColor: "text-white"
        };
    }
  };

  const theme = getThemeColors();

  const handleMpesaPayment = async () => {
    if (!phoneNumber || phoneNumber.length < 10) {
      toast.error("Please enter a valid phone number");
      return;
    }

    setIsProcessing(true);
    
    // Simulate API call
    setTimeout(() => {
      setIsProcessing(false);
      toast.success("M-Pesa payment initiated! Check your phone for the payment prompt.");
      onPaymentSuccess();
      onClose();
    }, 2000);
  };

  const handleCardPayment = async () => {
    if (!cardNumber || !expiryDate || !cvv) {
      toast.error("Please fill in all card details");
      return;
    }

    setIsProcessing(true);
    
    // Simulate API call
    setTimeout(() => {
      setIsProcessing(false);
      toast.success("Payment successful! Course unlocked.");
      onPaymentSuccess();
      onClose();
    }, 2000);
  };

  const handlePayment = () => {
    if (paymentMethod === "mpesa") {
      handleMpesaPayment();
    } else {
      handleCardPayment();
    }
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-md">
        <DialogHeader className={`-mx-6 -mt-6 px-6 pt-6 pb-4 bg-gradient-to-r ${theme.headerGradient} ${theme.textColor} rounded-t-lg`}>
          <DialogTitle className="flex items-center gap-2">
            <DollarSign size={24} />
            {classType === "crib" ? "🎨 Unlock Magic Adventure" :
             classType === "clicker" ? "🚀 Unlock Power-Up" :
             "Unlock Course"}
          </DialogTitle>
          <DialogDescription className={classType === "crib" ? "text-navy/90" : "text-white/90"}>
            {classType === "crib" ? "Complete payment to unlock this magical learning adventure!" :
             classType === "clicker" ? "Complete payment to unlock this awesome course and level up your skills!" :
             "Complete your payment to unlock full access to this course and start learning immediately."}
          </DialogDescription>
        </DialogHeader>

        <div className="mt-4">
          <Card className={`mb-4 border-2 ${theme.border}`}>
            <CardHeader className={`pb-3 bg-gradient-to-r ${theme.headerGradient} ${theme.textColor}`}>
              <h3 className="font-medium">{courseTitle}</h3>
              <div className="flex items-center justify-between">
                <span className="text-2xl font-bold">${price}</span>
                <Badge variant="secondary" className={`${theme.accent} ${classType === "crib" ? "text-navy" : "text-white"}`}>
                  {classType === "crib" ? "⭐ One payment" : "One-time payment"}
                </Badge>
              </div>
            </CardHeader>
            <CardContent className="pt-0">
              <div className="space-y-2 text-sm text-gray-600">
                {sessionQuantity && sessionDuration ? (
                  // Mentor session payment
                  <>
                    <div className="flex items-center gap-2">
                      <Check size={16} className={theme.icon} />
                      <span>{sessionQuantity} {sessionQuantity === 1 ? 'session' : 'sessions'}</span>
                    </div>
                    <div className="flex items-center gap-2">
                      <Check size={16} className={theme.icon} />
                      <span>{sessionDuration} minutes per session</span>
                    </div>
                    <div className="flex items-center gap-2">
                      <Check size={16} className={theme.icon} />
                      <span>{classType === "crib" ? "Magic helper time!" : "One-on-one mentoring"}</span>
                    </div>
                    <div className="flex items-center gap-2">
                      <Check size={16} className={theme.icon} />
                      <span>{classType === "crib" ? "Special learning help" : "Personalized guidance"}</span>
                    </div>
                    {sessionQuantity >= 5 && (
                      <div className="flex items-center gap-2">
                        <Check size={16} className={theme.icon} />
                        <span className={`${theme.primary} font-medium`}>{classType === "crib" ? "🌟 Super saver bonus!" : "5% bulk discount applied!"}</span>
                      </div>
                    )}
                  </>
                ) : (
                  // Course payment
                  <>
                    <div className="flex items-center gap-2">
                      <Check size={16} className={theme.icon} />
                      <span>{classType === "crib" ? "12 fun lessons" : "12 interactive sessions"}</span>
                    </div>
                    <div className="flex items-center gap-2">
                      <Check size={16} className={theme.icon} />
                      <span>{classType === "crib" ? "Forever yours!" : "Lifetime access"}</span>
                    </div>
                    <div className="flex items-center gap-2">
                      <Check size={16} className={theme.icon} />
                      <span>{classType === "crib" ? "Shiny certificate!" : "Certificate upon completion"}</span>
                    </div>
                    <div className="flex items-center gap-2">
                      <Check size={16} className={theme.icon} />
                      <span>{classType === "crib" ? "Magic helper support" : "Technical mentor support"}</span>
                    </div>
                  </>
                )}
              </div>
            </CardContent>
          </Card>

          <Tabs value={paymentMethod} onValueChange={setPaymentMethod}>
            <TabsList className="grid w-full grid-cols-2">
              <TabsTrigger value="mpesa" className="flex items-center gap-2">
                <Smartphone size={16} />
                M-Pesa
              </TabsTrigger>
              <TabsTrigger value="card" className="flex items-center gap-2">
                <CreditCard size={16} />
                Card
              </TabsTrigger>
            </TabsList>
            
            <TabsContent value="mpesa" className="space-y-4 mt-4">
              <div>
                <Label htmlFor="phone">Phone Number</Label>
                <Input
                  id="phone"
                  placeholder="e.g., +254712345678"
                  value={phoneNumber}
                  onChange={(e) => setPhoneNumber(e.target.value)}
                  className="mt-1"
                />
                <p className="text-xs text-gray-500 mt-1">
                  You'll receive an M-Pesa prompt on this number
                </p>
              </div>
            </TabsContent>
            
            <TabsContent value="card" className="space-y-4 mt-4">
              <div>
                <Label htmlFor="cardNumber">Card Number</Label>
                <Input
                  id="cardNumber"
                  placeholder="1234 5678 9012 3456"
                  value={cardNumber}
                  onChange={(e) => setCardNumber(e.target.value)}
                  className="mt-1"
                />
              </div>
              
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="expiry">Expiry Date</Label>
                  <Input
                    id="expiry"
                    placeholder="MM/YY"
                    value={expiryDate}
                    onChange={(e) => setExpiryDate(e.target.value)}
                    className="mt-1"
                  />
                </div>
                <div>
                  <Label htmlFor="cvv">CVV</Label>
                  <Input
                    id="cvv"
                    placeholder="123"
                    value={cvv}
                    onChange={(e) => setCvv(e.target.value)}
                    className="mt-1"
                  />
                </div>
              </div>
            </TabsContent>
          </Tabs>
        </div>
        
        <div className="flex gap-2 mt-6 pt-4 border-t">
          <Button variant="outline" onClick={onClose} className="flex-1">
            Cancel
          </Button>
          <Button 
            onClick={handlePayment} 
            disabled={isProcessing}
            className={`flex-1 ${theme.bg} ${theme.hover} text-white`}
          >
            {isProcessing ? (classType === "crib" ? "⏳ Magic happening..." : "Processing...") : 
             classType === "crib" ? `🌟 Pay ${price}` : `Pay ${price}`}
          </Button>
        </div>
      </DialogContent>
    </Dialog>
  );
}