import { useState } from "react";
import { Card, CardContent, CardHeader } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Avatar, AvatarFallback, AvatarImage } from "./ui/avatar";
import { 
  Breadcrumb,
  BreadcrumbList,
  BreadcrumbItem,
  BreadcrumbLink,
  BreadcrumbSeparator,
  BreadcrumbPage,
} from "./ui/breadcrumb";
import { PaymentModal } from "./PaymentModal";
import { ClassNavbar } from "./ClassNavbar";
import { CribNavbar } from "./CribNavbar";
import { ClickerNavbar } from "./ClickerNavbar";
import { QuestNavbar } from "./QuestNavbar";
import { 
  ArrowLeft, 
  Star, 
  Clock, 
  Calendar, 
  DollarSign,
  CheckCircle,
  User,
  MapPin,
  Globe,
  Home
} from "lucide-react";
import FullCalendar from '@fullcalendar/react';
import dayGridPlugin from '@fullcalendar/daygrid';
import timeGridPlugin from '@fullcalendar/timegrid';
import interactionPlugin from '@fullcalendar/interaction';
import { toast } from "sonner@2.0.3";

export function MentorBooking({ 
  mentor, 
  onBack, 
  courseTitle = "Mentor Session", 
  classType = "quest",
  onNavigateToDashboard,
  onBackToClasses 
}) {
  const [selectedSlot, setSelectedSlot] = useState(null);
  const [bookingDetails, setBookingDetails] = useState(null);
  const [showPayment, setShowPayment] = useState(false);
  const [sessionType, setSessionType] = useState("video");
  const [duration, setDuration] = useState(60); // minutes
  const [sessionQuantity, setSessionQuantity] = useState(1);
  const [durationFilter, setDurationFilter] = useState("all");

  // Generate available time slots for the next 7 days
  const generateTimeSlots = () => {
    const slots = [];
    const today = new Date();
    const durations = [30, 60, 90, 120]; // Available session durations in minutes
    
    for (let day = 1; day <= 7; day++) {
      const date = new Date(today);
      date.setDate(today.getDate() + day);
      
      // Available hours: 9 AM to 6 PM
      for (let hour = 9; hour < 18; hour++) {
        // Generate slots for each duration
        durations.forEach(sessionDuration => {
          // Only create slot if there's enough time before end of day
          if (hour + (sessionDuration / 60) <= 18) {
            const startTime = new Date(date);
            startTime.setHours(hour, 0, 0, 0);
            
            const endTime = new Date(startTime);
            endTime.setMinutes(startTime.getMinutes() + sessionDuration);
            
            // Random availability (80% chance)
            const isAvailable = Math.random() > 0.2;
            
            slots.push({
              id: `${date.toISOString().split('T')[0]}-${hour}-${sessionDuration}`,
              start: startTime.toISOString(),
              end: endTime.toISOString(),
              title: isAvailable ? `${sessionDuration}min Available` : `${sessionDuration}min Booked`,
              available: isAvailable,
              duration: sessionDuration
            });
          }
        });
      }
    }
    
    return slots;
  };

  const [timeSlots] = useState(generateTimeSlots());

  // Filter time slots based on duration filter
  const filteredTimeSlots = timeSlots.filter(slot => {
    if (durationFilter === "all") return true;
    return slot.duration === durationFilter;
  });

  const calendarEvents = filteredTimeSlots.map(slot => ({
    id: slot.id,
    start: slot.start,
    end: slot.end,
    title: slot.available ? `✓ ${slot.duration}min Available` : `✗ ${slot.duration}min Booked`,
    backgroundColor: slot.available ? '#38aae1' : '#dc2626',
    borderColor: slot.available ? '#38aae1' : '#dc2626',
    textColor: 'white',
    extendedProps: {
      available: slot.available,
      duration: slot.duration
    }
  }));

  const handleDateClick = (info) => {
    if (!info.event.extendedProps.available) {
      toast.error("This time slot is not available");
      return;
    }

    const slot = timeSlots.find(s => s.id === info.event.id);
    if (slot) {
      setSelectedSlot(slot);
      // Auto-set duration based on selected slot
      if (slot.duration) {
        setDuration(slot.duration);
      }
    }
  };

  const handleBookSession = () => {
    if (!selectedSlot) return;

    const slotDuration = selectedSlot.duration || duration;
    const singleSessionCost = (slotDuration / 60) * mentor.hourlyRate;
    const totalBeforeDiscount = singleSessionCost * sessionQuantity;
    const discount = sessionQuantity >= 5 ? 0.05 : 0;
    const totalCost = totalBeforeDiscount * (1 - discount);
    
    const booking = {
      date: new Date(selectedSlot.start).toLocaleDateString(),
      time: new Date(selectedSlot.start).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' }),
      duration: slotDuration,
      sessionType,
      sessionQuantity,
      totalCost
    };

    setBookingDetails(booking);
    setShowPayment(true);
  };

  const handlePaymentSuccess = () => {
    const sessionText = sessionQuantity === 1 ? "Session" : `${sessionQuantity} sessions`;
    toast.success(`${sessionText} booked successfully! You'll receive a confirmation email shortly.`);
    setShowPayment(false);
    onBack();
  };

  // Extended mentor info for booking page
  const extendedMentorInfo = {
    ...mentor,
    rating: 4.9,
    totalSessions: 127,
    responseTime: "< 5 min",
    bio: "Experienced coding instructor with 5+ years teaching programming to kids and teens. Specializes in visual programming languages and creative coding projects.",
    languages: ["English", "Spanish"],
    timezone: "EST (UTC-5)",
    avatar: undefined
  };

  // Age-appropriate theme classes
  const getThemeClasses = () => {
    switch (classType) {
      case "crib":
        return {
          gradient: "from-yellow via-light-blue/80 to-navy/60",
          navbarGradient: "from-yellow via-light-blue to-navy",
          accent: "text-light-blue",
          bg: "bg-gradient-to-r from-yellow via-light-blue to-navy"
        };
      case "clicker":
        return {
          gradient: "from-light-blue to-navy",
          navbarGradient: "from-light-blue to-navy",
          accent: "text-yellow",
          bg: "bg-gradient-to-r from-light-blue to-navy"
        };
      case "quest":
        return {
          gradient: "from-navy via-navy/90 to-navy/80",
          navbarGradient: "bg-navy",
          accent: "text-navy",
          bg: "bg-navy"
        };
    }
  };

  const theme = getThemeClasses();

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Class Navbar - Use specialized navbar for each class type */}
      {classType === "crib" ? (
        <CribNavbar 
          currentCourse={courseTitle}
          onBackToClasses={onBackToClasses || onBack}
          onNavigateToDashboard={onNavigateToDashboard}
          onHelp={() => {}}
          onContactUs={() => {}}
        />
      ) : classType === "clicker" ? (
        <ClickerNavbar 
          currentCourse={courseTitle}
          onBackToClasses={onBackToClasses || onBack}
          onNavigateToDashboard={onNavigateToDashboard}
          onHelp={() => {}}
          onContactUs={() => {}}
        />
      ) : classType === "quest" ? (
        <QuestNavbar 
          currentCourse={courseTitle}
          onBackToClasses={onBackToClasses || onBack}
          onNavigateToDashboard={onNavigateToDashboard}
          onHelp={() => {}}
          onContactUs={() => {}}
        />
      ) : (
        <ClassNavbar 
          classType={classType}
          currentCourse={courseTitle}
          onBackToClasses={onBackToClasses || onBack}
          onNavigateToDashboard={onNavigateToDashboard}
          onHelp={() => {}}
          onContactUs={() => {}}
        />
      )}

      {/* Breadcrumb Header */}
      <div className="bg-white border-b p-4">
        <div className="container mx-auto">
          <Breadcrumb>
            <BreadcrumbList>
              <BreadcrumbItem>
                <BreadcrumbLink 
                  onClick={onNavigateToDashboard}
                  className="flex items-center gap-1 cursor-pointer"
                >
                  <Home size={16} />
                  Dashboard
                </BreadcrumbLink>
              </BreadcrumbItem>
              <BreadcrumbSeparator />
              <BreadcrumbItem>
                <BreadcrumbLink 
                  onClick={onBackToClasses || onBack}
                  className="cursor-pointer"
                >
                  {classType === "crib" ? "Magic Classes" :
                   classType === "clicker" ? "Tech Academy" :
                   "Professional Courses"}
                </BreadcrumbLink>
              </BreadcrumbItem>
              <BreadcrumbSeparator />
              <BreadcrumbItem>
                <BreadcrumbLink 
                  onClick={onBack}
                  className="cursor-pointer"
                >
                  {courseTitle}
                </BreadcrumbLink>
              </BreadcrumbItem>
              <BreadcrumbSeparator />
              <BreadcrumbItem>
                <BreadcrumbPage className="font-medium">
                  Book Mentor Session
                </BreadcrumbPage>
              </BreadcrumbItem>
            </BreadcrumbList>
          </Breadcrumb>
        </div>
      </div>

      {/* Header */}
      <div className="p-6 bg-white">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <Button
              variant="outline"
              size="sm"
              onClick={onBack}
              className="border-gray-300 text-gray-700 hover:bg-gray-50"
            >
              <ArrowLeft size={16} className="mr-1" />
              Back to Lesson
            </Button>
          </div>
          
          <Badge variant="secondary" className="bg-gray-100 text-gray-700">
            {classType === "crib" ? "🌟 Book Magic Helper" :
             classType === "clicker" ? "🚀 Book Tech Mentor" :
             "📚 Book Professional Mentor"}
          </Badge>
        </div>
        
        <div className="mt-4">
          <h1 className="text-2xl mb-2 text-gray-900">
            {classType === "crib" ? `Get Magic Help from ${mentor.name}` :
             classType === "clicker" ? `Level Up with ${mentor.name}` :
             `Book a Session with ${mentor.name}`}
          </h1>
          <p className="text-gray-600">
            {classType === "crib" ? "Pick a time to learn magical coding tricks!" :
             classType === "clicker" ? "Choose your power-up session time and type" :
             "Choose an available time slot and session type"}
          </p>
        </div>
      </div>

      <div className="container mx-auto p-6">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Mentor Info Sidebar */}
          <div className="lg:col-span-1">
            <Card>
              <CardHeader>
                <div className="flex items-start gap-3">
                  <Avatar className="h-16 w-16">
                    <AvatarImage src={extendedMentorInfo.avatar} />
                    <AvatarFallback className="bg-light-blue text-white text-lg">
                      {mentor.name.split(' ').map(n => n[0]).join('')}
                    </AvatarFallback>
                  </Avatar>
                  
                  <div className="flex-1">
                    <h3 className="font-medium text-lg">{mentor.name}</h3>
                    <div className="flex items-center gap-2 mt-1">
                      <div className="flex items-center gap-1">
                        <Star className="text-yellow-400 fill-current" size={16} />
                        <span className="text-sm">{extendedMentorInfo.rating}</span>
                      </div>
                      <span className="text-sm text-gray-500">({extendedMentorInfo.totalSessions} sessions)</span>
                    </div>
                    <Badge variant="secondary" className="bg-green-100 text-green-700 mt-2">
                      Online
                    </Badge>
                  </div>
                </div>
              </CardHeader>
              
              <CardContent>
                <div className="space-y-4">
                  <div>
                    <h4 className="font-medium mb-2">Expertise</h4>
                    <div className="flex flex-wrap gap-1">
                      {mentor.expertise.map((skill) => (
                        <Badge key={skill} variant="outline" className="text-xs">
                          {skill}
                        </Badge>
                      ))}
                    </div>
                  </div>
                  
                  <div>
                    <h4 className="font-medium mb-2">About</h4>
                    <p className="text-sm text-gray-600">{extendedMentorInfo.bio}</p>
                  </div>
                  
                  <div className="space-y-2">
                    <div className="flex items-center gap-2 text-sm">
                      <Clock size={14} className="text-gray-400" />
                      <span>Responds {extendedMentorInfo.responseTime}</span>
                    </div>
                    <div className="flex items-center gap-2 text-sm">
                      <DollarSign size={14} className="text-gray-400" />
                      <span>${mentor.hourlyRate}/hour</span>
                    </div>
                    <div className="flex items-center gap-2 text-sm">
                      <Globe size={14} className="text-gray-400" />
                      <span>{extendedMentorInfo.timezone}</span>
                    </div>
                    <div className="flex items-center gap-2 text-sm">
                      <MapPin size={14} className="text-gray-400" />
                      <span>{extendedMentorInfo.languages.join(", ")}</span>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Session Configuration */}
            <Card className="mt-6">
              <CardHeader>
                <h3>Session Details</h3>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  <div>
                    <label className="text-sm font-medium mb-2 block">Session Type</label>
                    <div className="space-y-2">
                      {[
                        { value: "chat", label: "Text Chat", description: "Written conversation" },
                        { value: "video", label: "Video Call", description: "Face-to-face session" },
                        { value: "screen-share", label: "Screen Share", description: "Code together in real-time" }
                      ].map((type) => (
                        <label key={type.value} className="flex items-start gap-3 cursor-pointer">
                          <input
                            type="radio"
                            name="sessionType"
                            value={type.value}
                            checked={sessionType === type.value}
                            onChange={(e) => setSessionType(e.target.value)}
                            className="mt-1"
                          />
                          <div>
                            <div className="font-medium text-sm">{type.label}</div>
                            <div className="text-xs text-gray-600">{type.description}</div>
                          </div>
                        </label>
                      ))}
                    </div>
                  </div>
                  
                  <div>
                    <label className="text-sm font-medium mb-2 block">Duration</label>
                    <select
                      value={duration}
                      onChange={(e) => {
                        const newDuration = Number(e.target.value);
                        setDuration(newDuration);
                        setDurationFilter(newDuration); // Sync filter with duration selection
                      }}
                      className="w-full p-2 border rounded-md text-sm"
                    >
                      <option value={30}>30 minutes - ${(30/60 * mentor.hourlyRate).toFixed(0)}</option>
                      <option value={60}>1 hour - ${mentor.hourlyRate}</option>
                      <option value={90}>1.5 hours - ${(90/60 * mentor.hourlyRate).toFixed(0)}</option>
                      <option value={120}>2 hours - ${(120/60 * mentor.hourlyRate).toFixed(0)}</option>
                    </select>
                  </div>

                  <div>
                    <div className="flex items-center justify-between mb-2">
                      <label className="text-sm font-medium">Number of Sessions</label>
                      {sessionQuantity >= 5 && (
                        <Badge className="bg-green-100 text-green-700 text-xs">
                          💰 5% OFF
                        </Badge>
                      )}
                    </div>
                    <select
                      value={sessionQuantity}
                      onChange={(e) => setSessionQuantity(Number(e.target.value))}
                      className="w-full p-2 border rounded-md text-sm"
                    >
                      <option value={1}>1 Session - ${((duration / 60) * mentor.hourlyRate).toFixed(0)}</option>
                      <option value={2}>2 Sessions - ${((duration / 60) * mentor.hourlyRate * 2).toFixed(0)}</option>
                      <option value={3}>3 Sessions - ${((duration / 60) * mentor.hourlyRate * 3).toFixed(0)}</option>
                      <option value={4}>4 Sessions - ${((duration / 60) * mentor.hourlyRate * 4).toFixed(0)}</option>
                      <option value={5}>5 Sessions - ${(((duration / 60) * mentor.hourlyRate * 5) * 0.95).toFixed(0)} (Was ${((duration / 60) * mentor.hourlyRate * 5).toFixed(0)})</option>
                    </select>
                    {sessionQuantity >= 5 && (
                      <div className="mt-2 p-2 bg-green-50 rounded-md border border-green-200">
                        <p className="text-xs text-green-700 flex items-center gap-1">
                          <CheckCircle size={12} />
                          <span className="font-medium">Bulk Discount Applied!</span>
                          Save ${(((duration / 60) * mentor.hourlyRate * sessionQuantity) * 0.05).toFixed(0)} with 5+ sessions
                        </p>
                      </div>
                    )}
                    <p className="text-xs text-gray-500 mt-1">
                      💡 Book multiple sessions to save time scheduling future meetings
                    </p>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Calendar */}
          <div className="lg:col-span-2">
            <Card>
              <CardHeader>
                <h3>Available Time Slots</h3>
                <p className="text-sm text-gray-600">
                  Click on an available time slot to book your session
                </p>
              </CardHeader>
              <CardContent>
                {/* Duration Filter */}
                <div className="mb-4 p-4 bg-gray-50 rounded-lg border">
                  <div className="flex items-center justify-between mb-3">
                    <label className="text-sm font-medium">Filter by Session Duration</label>
                    <div className="flex items-center gap-2">
                      <Badge variant="outline" className="text-xs">
                        {filteredTimeSlots.filter(slot => slot.available).length} available slots
                      </Badge>
                      <Badge variant="outline" className="text-xs">
                        {durationFilter === "all" ? "All Durations" : `${durationFilter} min sessions`}
                      </Badge>
                    </div>
                  </div>
                  <div className="flex flex-wrap gap-2">
                    <Button
                      variant={durationFilter === "all" ? "default" : "outline"}
                      size="sm"
                      onClick={() => setDurationFilter("all")}
                      className="text-xs"
                    >
                      All Sessions
                    </Button>
                    <Button
                      variant={durationFilter === 30 ? "default" : "outline"}
                      size="sm"
                      onClick={() => setDurationFilter(30)}
                      className="text-xs"
                    >
                      30 min
                    </Button>
                    <Button
                      variant={durationFilter === 60 ? "default" : "outline"}
                      size="sm"
                      onClick={() => setDurationFilter(60)}
                      className="text-xs"
                    >
                      1 hour
                    </Button>
                    <Button
                      variant={durationFilter === 90 ? "default" : "outline"}
                      size="sm"
                      onClick={() => setDurationFilter(90)}
                      className="text-xs"
                    >
                      1.5 hours
                    </Button>
                    <Button
                      variant={durationFilter === 120 ? "default" : "outline"}
                      size="sm"
                      onClick={() => setDurationFilter(120)}
                      className="text-xs"
                    >
                      2 hours
                    </Button>
                  </div>
                  <p className="text-xs text-gray-500 mt-2">
                    💡 Filter shows only slots that can accommodate your selected duration
                  </p>
                </div>

                <div className="h-96 mb-4">
                  <FullCalendar
                    plugins={[dayGridPlugin, timeGridPlugin, interactionPlugin]}
                    initialView="timeGridWeek"
                    headerToolbar={{
                      left: 'prev,next today',
                      center: 'title',
                      right: 'dayGridMonth,timeGridWeek,timeGridDay'
                    }}
                    events={calendarEvents}
                    eventClick={handleDateClick}
                    height="100%"
                    slotMinTime="09:00:00"
                    slotMaxTime="18:00:00"
                    weekends={true}
                    selectable={false}
                    eventDisplay="block"
                  />
                </div>
                
                {selectedSlot && (
                  <Card className="mt-4 border-light-blue bg-light-blue/5">
                    <CardContent className="p-4">
                      <div className="flex items-center justify-between">
                        <div>
                          <div className="flex items-center gap-2 mb-2">
                            <CheckCircle className="text-light-blue" size={20} />
                            <h4 className="font-medium">Selected Time Slot</h4>
                          </div>
                          <p className="text-sm text-gray-600">
                            {new Date(selectedSlot.start).toLocaleDateString()} at{' '}
                            {new Date(selectedSlot.start).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
                          </p>
                          <p className="text-sm text-gray-600">
                            Duration: {selectedSlot.duration || duration} minutes • Type: {sessionType.charAt(0).toUpperCase() + sessionType.slice(1)}
                          </p>
                          <p className="text-sm text-gray-600">
                            {sessionQuantity} {sessionQuantity === 1 ? 'Session' : 'Sessions'} 
                            {sessionQuantity >= 5 && <span className="text-green-600 ml-1">(5% discount applied)</span>}
                          </p>
                          <p className="text-sm font-medium mt-1">
                            Total Cost: ${(() => {
                              const slotDuration = selectedSlot.duration || duration;
                              const singleSessionCost = (slotDuration / 60) * mentor.hourlyRate;
                              const totalBeforeDiscount = singleSessionCost * sessionQuantity;
                              const discount = sessionQuantity >= 5 ? 0.05 : 0;
                              return (totalBeforeDiscount * (1 - discount)).toFixed(0);
                            })()}
                          </p>
                        </div>
                        <Button
                          onClick={handleBookSession}
                          className="bg-light-blue hover:bg-light-blue/90"
                        >
                          <Calendar size={16} className="mr-2" />
                          Book Session
                        </Button>
                      </div>
                    </CardContent>
                  </Card>
                )}
              </CardContent>
            </Card>
          </div>
        </div>
      </div>

      {/* Payment Modal */}
      {showPayment && bookingDetails && (
        <PaymentModal
          isOpen={showPayment}
          onClose={() => setShowPayment(false)}
          onPaymentSuccess={handlePaymentSuccess}
          courseTitle={`${bookingDetails.sessionQuantity > 1 ? `${bookingDetails.sessionQuantity} sessions` : 'Session'} with ${mentor.name}`}
          price={bookingDetails.totalCost}
          sessionQuantity={bookingDetails.sessionQuantity}
          sessionDuration={bookingDetails.duration}
        />
      )}
    </div>
  );
}