import React from "react";
import { Card, CardContent, CardHeader } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { Progress } from "./ui/progress";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { 
  Breadcrumb,
  BreadcrumbList,
  BreadcrumbItem,
  BreadcrumbLink,
  BreadcrumbSeparator,
  BreadcrumbPage,
} from "./ui/breadcrumb";
import { 
  ArrowLeft, 
  Play, 
  CheckCircle, 
  Circle, 
  FileText, 
  Video, 
  HelpCircle,
  ChevronRight,
  ChevronLeft,
  Upload,
  ExternalLink,
  Youtube,
  File,
  Presentation,
  Clock,
  Home
} from "lucide-react";
import companyLogo from "figma:asset/517ee9d784589cb130eb229f0ac854531acfb4ba.png";
import { VideoPlayer } from "./VideoPlayer";
import { SupportModal } from "./SupportModal";
import { PictobloxEmbed } from "./PictobloxEmbed";
import { ClassNavbar } from "./ClassNavbar";
import { CribNavbar } from "./CribNavbar";
import { ClickerNavbar } from "./ClickerNavbar";
import { QuestNavbar } from "./QuestNavbar";
import { toast } from "sonner@2.0.3";

export class LessonView extends React.Component {
  constructor(props) {
    super(props);
    
    this.state = {
      activeTab: "videos",
      activeVideoType: "introduction",
      isSupportModalOpen: false,
      isPictobloxVisible: false,
      uploadedFile: null,
      sessionTime: 0,
      totalCourseTime: 0,
      checklistCompleted: {}
    };

    this.startTimeRef = Date.now();
    this.intervalRef = null;

    // Bind methods
    this.handleFileUpload = this.handleFileUpload.bind(this);
    this.formatTime = this.formatTime.bind(this);
    this.saveTimeToStorage = this.saveTimeToStorage.bind(this);
    this.loadTimeFromStorage = this.loadTimeFromStorage.bind(this);
    this.getCurrentVideo = this.getCurrentVideo.bind(this);
    this.getThemeClasses = this.getThemeClasses.bind(this);
    this.getSessionTitles = this.getSessionTitles.bind(this);
    this.getVideoTypes = this.getVideoTypes.bind(this);
    this.getResources = this.getResources.bind(this);
    this.getChecklist = this.getChecklist.bind(this);
    this.toggleChecklistItem = this.toggleChecklistItem.bind(this);
  }

  componentDidMount() {
    this.loadTimeFromStorage();
    this.startTimeRef = Date.now();

    // Start interval to update session time
    this.intervalRef = setInterval(() => {
      this.setState(prevState => {
        const newSessionTime = prevState.sessionTime + 1;
        // Save every 30 seconds to avoid too frequent localStorage writes
        if (newSessionTime % 30 === 0) {
          const newTotal = prevState.totalCourseTime + 1;
          this.saveTimeToStorage(newSessionTime, newTotal);
          return {
            sessionTime: newSessionTime,
            totalCourseTime: newTotal
          };
        }
        return { sessionTime: newSessionTime };
      });
    }, 1000);
  }

  componentDidUpdate(prevProps) {
    // Re-initialize time tracking when session or class type changes
    if (prevProps.currentSession !== this.props.currentSession || 
        prevProps.classType !== this.props.classType) {
      // Save final time before changing
      if (this.intervalRef) {
        clearInterval(this.intervalRef);
        const finalSessionTime = this.state.sessionTime + Math.floor((Date.now() - this.startTimeRef) / 1000);
        const finalTotalTime = this.state.totalCourseTime + Math.floor((Date.now() - this.startTimeRef) / 1000);
        this.saveTimeToStorage(finalSessionTime, finalTotalTime);
      }

      // Reload for new session
      this.loadTimeFromStorage();
      this.startTimeRef = Date.now();

      // Restart interval
      this.intervalRef = setInterval(() => {
        this.setState(prevState => {
          const newSessionTime = prevState.sessionTime + 1;
          if (newSessionTime % 30 === 0) {
            const newTotal = prevState.totalCourseTime + 1;
            this.saveTimeToStorage(newSessionTime, newTotal);
            return {
              sessionTime: newSessionTime,
              totalCourseTime: newTotal
            };
          }
          return { sessionTime: newSessionTime };
        });
      }, 1000);
    }
  }

  componentWillUnmount() {
    // Save final time when component unmounts
    if (this.intervalRef) {
      clearInterval(this.intervalRef);
      const finalSessionTime = this.state.sessionTime + Math.floor((Date.now() - this.startTimeRef) / 1000);
      const finalTotalTime = this.state.totalCourseTime + Math.floor((Date.now() - this.startTimeRef) / 1000);
      this.saveTimeToStorage(finalSessionTime, finalTotalTime);
    }
  }

  getSessionTitles() {
    const { classType, totalSessions } = this.props;
    
    switch (classType) {
      case "crib":
        return [
          "🌟 Welcome to Magic Land",
          "🎨 Meet Your Colorful Friends", 
          "🎮 First Magic Spell",
          "🌈 Rainbow Adventures",
          "🎭 Story Time Fun",
          "🎪 Circus of Colors",
          "🦄 Unicorn Helpers",
          "🌸 Flower Power Magic",
          "🎯 Target Practice Game",
          "🎨 Art Gallery Show",
          "🎪 Grand Finale Show",
          "🏆 Magic Master Certificate"
        ];
      case "clicker":
        return [
          "🚀 Mission Launch",
          "⚡ Power-Up Basics",
          "🎮 Game Engine Intro",
          "🔧 Build Your Toolkit",
          "💻 Code Like a Pro",
          "🎯 Target Achievement",
          "🔥 Advanced Techniques",
          "🏗️ Construction Zone",
          "🎪 Feature Factory",
          "🚀 Deploy & Test",
          "🏆 Champion Challenge",
          "💎 Master Developer",
          "🌟 Elite Certification",
          "🎖️ Hero Badge",
          "👑 Tech Royalty"
        ];
      case "quest":
        return [
          "Foundation Setup",
          "Core Principles",
          "Development Environment",
          "Algorithm Design",
          "Implementation Phase",
          "Testing & Debugging",
          "Performance Optimization",
          "Security Implementation",
          "Database Integration",
          "API Development",
          "Frontend Integration",
          "Deployment Pipeline",
          "Monitoring & Analytics",
          "Production Release",
          "Portfolio Integration",
          "Professional Certification"
        ];
      default:
        return Array.from({ length: totalSessions }, (_, i) => 
          i === 0 ? "Getting Started" : 
          i === 1 ? "Basic Concepts" :
          i === 2 ? "Hands-on Practice" :
          i === 3 ? "Creative Challenge" :
          i === 4 ? "Project Building" :
          `Session ${i + 1}`
        );
    }
  }

  getVideoTypes() {
    const { classType } = this.props;
    
    switch (classType) {
      case "crib":
        return [
          { id: "introduction", label: "🎬 Story Time", icon: Play },
          { id: "activity", label: "🎮 Fun Games", icon: Video },
          { id: "assignment", label: "🎨 Show & Tell", icon: FileText }
        ];
      case "clicker":
        return [
          { id: "introduction", label: "🚀 Mission Brief", icon: Play },
          { id: "activity", label: "⚡ Action Labs", icon: Video },
          { id: "assignment", label: "🏆 Challenge Quest", icon: FileText }
        ];
      case "quest":
        return [
          { id: "introduction", label: "📋 Module Overview", icon: Play },
          { id: "activity", label: "💻 Practical Labs", icon: Video },
          { id: "assignment", label: "📊 Project Submission", icon: FileText }
        ];
      default:
        return [
          { id: "introduction", label: "Introduction", icon: Play },
          { id: "activity", label: "Activities", icon: Video },
          { id: "assignment", label: "Assignment", icon: FileText }
        ];
    }
  }

  getResources() {
    const { classType } = this.props;
    
    switch (classType) {
      case "crib":
        return [
          { 
            id: "1", 
            title: "🌈 My Colorful Learning Book", 
            type: "pdf", 
            url: "#",
            icon: File 
          },
          { 
            id: "2", 
            title: "🎭 Fun Learning Videos", 
            type: "youtube", 
            url: "#",
            icon: Youtube 
          },
          { 
            id: "3", 
            title: "🎨 Picture Story Slides", 
            type: "presentation", 
            url: "#",
            icon: Presentation 
          },
          { 
            id: "4", 
            title: "🌟 Magic Code Examples", 
            type: "docs", 
            url: "#",
            icon: FileText 
          }
        ];
      case "clicker":
        return [
          { 
            id: "1", 
            title: "🎯 Tech Explorer Guidebook", 
            type: "pdf", 
            url: "#",
            icon: File 
          },
          { 
            id: "2", 
            title: "⚡ Coding Adventure Videos", 
            type: "youtube", 
            url: "#",
            icon: Youtube 
          },
          { 
            id: "3", 
            title: "🚀 Mission Control Slides", 
            type: "presentation", 
            url: "#",
            icon: Presentation 
          },
          { 
            id: "4", 
            title: "💻 Code Arsenal & Examples", 
            type: "docs", 
            url: "#",
            icon: FileText 
          }
        ];
      case "quest":
        return [
          { 
            id: "1", 
            title: "📚 Technical Documentation", 
            type: "pdf", 
            url: "#",
            icon: File 
          },
          { 
            id: "2", 
            title: "🎥 Professional Tutorial Series", 
            type: "youtube", 
            url: "#",
            icon: Youtube 
          },
          { 
            id: "3", 
            title: "📊 Industry Standards Presentation", 
            type: "presentation", 
            url: "#",
            icon: Presentation 
          },
          { 
            id: "4", 
            title: "💼 Production Code Repository", 
            type: "docs", 
            url: "#",
            icon: FileText 
          }
        ];
      default:
        return [
          { 
            id: "1", 
            title: "Course Handbook (PDF)", 
            type: "pdf", 
            url: "#",
            icon: File 
          },
          { 
            id: "2", 
            title: "Tutorial Video Series", 
            type: "youtube", 
            url: "#",
            icon: Youtube 
          },
          { 
            id: "3", 
            title: "Course Presentation", 
            type: "presentation", 
            url: "#",
            icon: Presentation 
          },
          { 
            id: "4", 
            title: "Code Examples (Google Doc)", 
            type: "docs", 
            url: "#",
            icon: FileText 
          }
        ];
    }
  }

  getChecklist() {
    const { classType } = this.props;
    
    switch (classType) {
      case "crib":
        return [
          { id: 1, task: "🔌 Turn on your magical Quarky friend", completed: true },
          { id: 2, task: "🎮 Connect Quarky to your colorful PictoBlox", completed: true },
          { id: 3, task: "🌟 Pick a fun magic power for Quarky", completed: false },
          { id: 4, task: "🎨 Create your magical code blocks", completed: false },
          { id: 5, task: "🎭 Watch your creation come to life!", completed: false }
        ];
      case "clicker":
        return [
          { id: 1, task: "⚡ Power up Quarky robot", completed: true },
          { id: 2, task: "🔗 Link Quarky to PictoBlox via Bluetooth", completed: true },
          { id: 3, task: "🤖 Choose your AI superpower", completed: false },
          { id: 4, task: "💻 Build your awesome code", completed: false },
          { id: 5, task: "🚀 Test and debug your creation", completed: false }
        ];
      case "quest":
        return [
          { id: 1, task: "Initialize Quarky hardware module", completed: true },
          { id: 2, task: "Establish Bluetooth/Serial connection to PictoBlox", completed: true },
          { id: 3, task: "Configure AI model parameters", completed: false },
          { id: 4, task: "Implement algorithm logic", completed: false },
          { id: 5, task: "Execute comprehensive testing protocol", completed: false }
        ];
      default:
        return [
          { id: 1, task: "Switch Quarky on", completed: true },
          { id: 2, task: "Connect Quarky to PictoBlox using Bluetooth or Serial port", completed: true },
          { id: 3, task: "Select AI model to use", completed: false },
          { id: 4, task: "Write your code", completed: false },
          { id: 5, task: "Test Functionality", completed: false }
        ];
    }
  }

  formatTime(seconds) {
    const hours = Math.floor(seconds / 3600);
    const mins = Math.floor((seconds % 3600) / 60);
    const secs = seconds % 60;
    
    if (hours > 0) {
      return `${hours}h ${mins}m ${secs}s`;
    } else if (mins > 0) {
      return `${mins}m ${secs}s`;
    } else {
      return `${secs}s`;
    }
  }

  saveTimeToStorage(sessionTime, totalTime) {
    const { classType, currentSession } = this.props;
    const timeData = {
      courseId: `${classType}-course`,
      sessionTime: sessionTime,
      totalCourseTime: totalTime,
      lastUpdated: Date.now()
    };
    localStorage.setItem(`lesson_time_${classType}_${currentSession}`, JSON.stringify(timeData));
  }

  loadTimeFromStorage() {
    const { classType, currentSession, totalSessions } = this.props;
    
    try {
      const savedData = localStorage.getItem(`lesson_time_${classType}_${currentSession}`);
      if (savedData) {
        const timeData = JSON.parse(savedData);
        this.setState({ sessionTime: timeData.sessionTime || 0 });
      }

      // Load total course time from all sessions
      let totalTime = 0;
      for (let i = 1; i <= totalSessions; i++) {
        const sessionData = localStorage.getItem(`lesson_time_${classType}_${i}`);
        if (sessionData) {
          const data = JSON.parse(sessionData);
          totalTime += data.sessionTime || 0;
        }
      }
      this.setState({ totalCourseTime: totalTime });
    } catch (error) {
      console.error('Error loading time data:', error);
    }
  }

  handleFileUpload(event) {
    const file = event.target.files?.[0];
    if (file) {
      this.setState({ uploadedFile: file });
      toast.success(`File "${file.name}" uploaded successfully!`);
    }
  }

  getCurrentVideo() {
    const { currentSession } = this.props;
    const { activeVideoType } = this.state;
    const sessions = this.getSessions();
    const currentSessionTitle = sessions[currentSession - 1]?.title;
    
    const getVideoContent = () => {
      const { classType } = this.props;
      
      switch (classType) {
        case "crib":
          return {
            introduction: {
              title: `${currentSessionTitle} - Story Time`,
              description: "🎬 Let's hear the magical story for today's adventure!"
            },
            activity: {
              title: `${currentSessionTitle} - Fun Games`,
              description: "🎮 Time to play and practice with colorful activities!"
            },
            assignment: {
              title: `${currentSessionTitle} - Show & Tell`,
              description: "🎨 Create something amazing to show your friends!"
            }
          };
        case "clicker":
          return {
            introduction: {
              title: `${currentSessionTitle} - Mission Brief`,
              description: "🚀 Get ready for today's epic coding challenge!"
            },
            activity: {
              title: `${currentSessionTitle} - Action Labs`,
              description: "⚡ Build, test, and level up your tech skills!"
            },
            assignment: {
              title: `${currentSessionTitle} - Challenge Quest`,
              description: "🏆 Complete the mission to unlock your next achievement!"
            }
          };
        case "quest":
          return {
            introduction: {
              title: `${currentSessionTitle} - Module Overview`,
              description: "📋 Professional concepts and industry best practices"
            },
            activity: {
              title: `${currentSessionTitle} - Practical Labs`,
              description: "💻 Hands-on development with real-world applications"
            },
            assignment: {
              title: `${currentSessionTitle} - Project Submission`,
              description: "📊 Demonstrate mastery through portfolio-quality work"
            }
          };
        default:
          return {
            introduction: {
              title: `${currentSessionTitle} - Introduction`,
              description: "Learn the key concepts for this session"
            },
            activity: {
              title: `${currentSessionTitle} - Activities`, 
              description: "Practice what you've learned with interactive exercises"
            },
            assignment: {
              title: `${currentSessionTitle} - Assignment`,
              description: "Complete the assignment to test your understanding"
            }
          };
      }
    };
    
    return getVideoContent()[activeVideoType];
  }

  getThemeClasses() {
    const { classType } = this.props;
    
    switch (classType) {
      case "crib":
        return {
          gradient: "from-light-blue via-light-blue/80 to-light-blue/60",
          navbarGradient: "from-light-blue via-light-blue/90 to-navy",
          accent: "text-light-blue",
          bg: "bg-gradient-to-r from-light-blue via-light-blue/90 to-navy"
        };
      case "clicker":
        return {
          gradient: "from-navy via-navy/80 to-navy/60",
          navbarGradient: "from-navy via-navy/90 to-light-blue",
          accent: "text-navy",
          bg: "bg-gradient-to-r from-navy via-navy/90 to-light-blue"
        };
      case "quest":
        return {
          gradient: "from-navy via-navy/80 to-light-blue/60",
          navbarGradient: "from-navy to-light-blue",
          accent: "text-navy",
          bg: "bg-gradient-to-r from-navy to-light-blue"
        };
    }
  }

  getSessions() {
    const { currentSession, totalSessions, classType } = this.props;
    const sessionTitles = this.getSessionTitles();
    
    return Array.from({ length: totalSessions }, (_, i) => ({
      id: `session-${i + 1}`,
      title: sessionTitles[i] || `Session ${i + 1}`,
      isCompleted: i < currentSession - 1,
      isLocked: i > currentSession - 1,
      duration: classType === "crib" ? "10 min" : 
                classType === "clicker" ? "15 min" : 
                "20 min"
    }));
  }

  toggleChecklistItem(id) {
    this.setState(prevState => {
      const updatedChecklistCompleted = {
        ...prevState.checklistCompleted,
        [id]: !prevState.checklistCompleted[id]
      };
      return { checklistCompleted: updatedChecklistCompleted };
    });
  }

  render() {
    const { 
      courseTitle, 
      currentSession, 
      totalSessions, 
      classType,
      onBack,
      onSessionComplete,
      onNavigateToSession,
      onSelectMentor,
      onNavigateToDashboard
    } = this.props;

    const {
      activeTab,
      activeVideoType,
      isSupportModalOpen,
      isPictobloxVisible,
      uploadedFile,
      sessionTime,
      totalCourseTime,
      checklistCompleted
    } = this.state;

    const sessions = this.getSessions();
    const videoTypes = this.getVideoTypes();
    const resources = this.getResources();
    const checklist = this.getChecklist();
    const theme = this.getThemeClasses();
    const progress = (currentSession / totalSessions) * 100;

    return (
      <div className="min-h-screen bg-gray-50">
        {/* Class Navbar - Use specialized navbar for each class type */}
        {classType === "crib" ? (
          <CribNavbar 
            currentCourse={courseTitle}
            onBackToClasses={onBack}
            onNavigateToDashboard={onNavigateToDashboard}
            onHelp={() => this.setState({ isSupportModalOpen: true })}
            onContactUs={() => this.setState({ isSupportModalOpen: true })}
          />
        ) : classType === "clicker" ? (
          <ClickerNavbar 
            currentCourse={courseTitle}
            onBackToClasses={onBack}
            onNavigateToDashboard={onNavigateToDashboard}
            onHelp={() => this.setState({ isSupportModalOpen: true })}
            onContactUs={() => this.setState({ isSupportModalOpen: true })}
          />
        ) : classType === "quest" ? (
          <QuestNavbar 
            currentCourse={courseTitle}
            onBackToClasses={onBack}
            onNavigateToDashboard={onNavigateToDashboard}
            onHelp={() => this.setState({ isSupportModalOpen: true })}
            onContactUs={() => this.setState({ isSupportModalOpen: true })}
          />
        ) : (
          <ClassNavbar 
            classType={classType}
            currentCourse={courseTitle}
            onBackToClasses={onBack}
            onNavigateToDashboard={onNavigateToDashboard}
            onHelp={() => this.setState({ isSupportModalOpen: true })}
            onContactUs={() => this.setState({ isSupportModalOpen: true })}
          />
        )}

        {/* Breadcrumb Header */}
        <div className="bg-white border-b p-4">
          <div className="container mx-auto">
            <Breadcrumb>
              <BreadcrumbList>
                <BreadcrumbItem>
                  <BreadcrumbLink 
                    onClick={onNavigateToDashboard}
                    className="flex items-center gap-1 cursor-pointer"
                  >
                    <Home size={16} />
                    Dashboard
                  </BreadcrumbLink>
                </BreadcrumbItem>
                <BreadcrumbSeparator />
                <BreadcrumbItem>
                  <BreadcrumbLink 
                    onClick={onBack}
                    className="cursor-pointer"
                  >
                    {classType === "crib" ? "Magic Classes" :
                     classType === "clicker" ? "Tech Academy" :
                     "Professional Courses"}
                  </BreadcrumbLink>
                </BreadcrumbItem>
                <BreadcrumbSeparator />
                <BreadcrumbItem>
                  <BreadcrumbPage className="font-medium">
                    {courseTitle}
                  </BreadcrumbPage>
                </BreadcrumbItem>
              </BreadcrumbList>
            </Breadcrumb>
          </div>
        </div>

        <div className="container mx-auto p-6">
          {/* Session Navigation and Pictoblox Toggle */}
          <div className="mb-4 flex justify-between items-center">
            {/* Previous/Next Navigation */}
            <div className="flex gap-3">
              <Button
                onClick={() => onNavigateToSession(currentSession - 1)}
                disabled={currentSession === 1}
                variant="outline"
                className={
                  classType === "crib"
                    ? "border-crib-teal text-crib-teal hover:bg-crib-teal hover:text-white disabled:opacity-50 disabled:cursor-not-allowed"
                    : classType === "clicker"
                    ? "border-navy text-navy hover:bg-navy hover:text-white disabled:opacity-50 disabled:cursor-not-allowed"
                    : "border-light-blue text-light-blue hover:bg-light-blue hover:text-white disabled:opacity-50 disabled:cursor-not-allowed"
                }
              >
                <ChevronLeft size={16} className="mr-1" />
                Previous
              </Button>
              
              <Button
                onClick={() => onNavigateToSession(currentSession + 1)}
                disabled={currentSession >= totalSessions}
                variant="outline"
                className={
                  classType === "crib"
                    ? "border-crib-teal text-crib-teal hover:bg-crib-teal hover:text-white disabled:opacity-50 disabled:cursor-not-allowed"
                    : classType === "clicker"
                    ? "border-navy text-navy hover:bg-navy hover:text-white disabled:opacity-50 disabled:cursor-not-allowed"
                    : "border-light-blue text-light-blue hover:bg-light-blue hover:text-white disabled:opacity-50 disabled:cursor-not-allowed"
                }
              >
                Next
                <ChevronRight size={16} className="ml-1" />
              </Button>
            </div>

            {/* Pictoblox Toggle Button */}
            <Button
              onClick={() => this.setState({ isPictobloxVisible: !isPictobloxVisible })}
              className={
                classType === "crib" 
                  ? "bg-gradient-to-r from-crib-teal via-crib-teal to-navy hover:from-crib-teal/90 hover:via-crib-teal/90 hover:to-navy/90" 
                  : classType === "clicker"
                  ? "bg-navy hover:bg-navy/90"
                  : "bg-light-blue hover:bg-light-blue/90"
              }
            >
              {isPictobloxVisible ? "Hide PictoBlox" : "Open PictoBlox"}
            </Button>
          </div>

          {/* Pictoblox Embed */}
          {isPictobloxVisible && (
            <div className="mb-6">
              <PictobloxEmbed sessionNumber={currentSession} />
            </div>
          )}

          <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
            {/* Main Content */}
            <div className="lg:col-span-2">
              <Card>
                <CardContent className="p-0">
                  {/* Video Type Selector */}
                  <div className="p-4 border-b bg-gray-50">
                    <div className="flex gap-2">
                      {videoTypes.map((type) => {
                        const Icon = type.icon;
                        return (
                          <Button
                            key={type.id}
                            variant={activeVideoType === type.id ? "default" : "outline"}
                            size="sm"
                            onClick={() => this.setState({ activeVideoType: type.id })}
                            className="flex items-center gap-2"
                          >
                            <Icon size={16} />
                            {type.label}
                          </Button>
                        );
                      })}
                    </div>
                  </div>

                  {/* Video Player */}
                  <VideoPlayer 
                    title={this.getCurrentVideo()?.title}
                    className="w-full"
                  />
                  
                  {/* Video Info */}
                  <div className="p-4 border-b">
                    <h3 className="font-medium mb-1">{this.getCurrentVideo()?.title}</h3>
                    <p className="text-sm text-gray-600">{this.getCurrentVideo()?.description}</p>
                  </div>

                  {/* Session Complete Action */}
                  <div className="p-4 flex justify-center">
                    <Button 
                      onClick={onSessionComplete}
                      className="bg-light-blue hover:bg-light-blue/90 text-white flex items-center gap-2"
                    >
                      {classType === "crib" ? "🎉 I Finished! Next Fun" :
                       classType === "clicker" ? "🏆 Mission Complete! Continue" :
                       "✅ Complete & Continue"}
                      <ChevronRight size={16} />
                    </Button>
                  </div>
                </CardContent>
              </Card>

              {/* PictoBlox Embed */}
              <div className="mt-6">
                <PictobloxEmbed sessionNumber={currentSession} />
              </div>

              {/* Session Checklist */}
              <Card className="mt-6">
                <CardHeader>
                  <h3 className="flex items-center gap-2">
                    <CheckCircle className="text-light-blue" size={20} />
                    {classType === "crib" ? "🌟 Magic To-Do List" :
                     classType === "clicker" ? "🎯 Mission Checklist" :
                     "📋 Session Requirements"}
                  </h3>
                  <p className="text-sm text-gray-600">
                    {classType === "crib" ? "Complete these magical steps for today!" :
                     classType === "clicker" ? "Complete these steps to win the mission!" :
                     "Complete these requirements for module certification"}
                  </p>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    {checklist.map((item) => {
                      const isCompleted = checklistCompleted[item.id] !== undefined 
                        ? checklistCompleted[item.id] 
                        : item.completed;
                      
                      return (
                        <div 
                          key={item.id} 
                          className="flex items-center gap-3 cursor-pointer hover:bg-gray-50 p-2 rounded-lg transition-colors"
                          onClick={() => this.toggleChecklistItem(item.id)}
                        >
                          {isCompleted ? (
                            <CheckCircle className="text-green-500 flex-shrink-0" size={16} />
                          ) : (
                            <Circle className="text-gray-400 flex-shrink-0" size={16} />
                          )}
                          <span className={isCompleted ? "text-gray-600 line-through" : "text-gray-900"}>
                            {item.task}
                          </span>
                        </div>
                      );
                    })}
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Sidebar */}
            <div className="space-y-6">
              {/* Session Navigation */}
              <Card className="overflow-hidden border-0 shadow-lg">
                <div className={`${theme.bg} text-white p-4`}>
                  {/* Course Info with Image */}
                  <div className="flex items-start gap-3 mb-4">
                    <div className="w-12 h-12 rounded-lg overflow-hidden flex-shrink-0 bg-white/20">
                      <img 
                        src="https://images.unsplash.com/photo-1565229284535-2cbbe3049123?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxwcm9ncmFtbWluZyUyMGNvZGluZyUyMGNvdXJzZSUyMGRpZ2l0YWwlMjBsZWFybmluZ3xlbnwxfHx8fDE3NTkxMjU1MDZ8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral"
                        alt={courseTitle}
                        className="w-full h-full object-cover"
                      />
                    </div>
                    <div className="flex-1 min-w-0">
                      <h3 className="font-semibold mb-1 truncate">
                        {courseTitle}
                      </h3>
                      <p className="text-white/80 text-sm">
                        {classType === "crib" ? "🌟 My Adventure" :
                         classType === "clicker" ? "🚀 Mission Progress" :
                         "📊 Module Progress"}
                      </p>
                    </div>
                  </div>

                  <div className="flex items-center justify-center mb-3">
                    <div className="flex items-center gap-2">
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => onNavigateToSession(currentSession - 1)}
                        disabled={currentSession === 1}
                        className="text-white hover:bg-white/20 h-8 w-8 p-0"
                      >
                        <ChevronLeft size={16} />
                      </Button>
                      <span className="text-white/90 text-sm px-2 min-w-[4rem] text-center">
                        {currentSession} / {totalSessions}
                      </span>
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => onNavigateToSession(currentSession + 1)}
                        disabled={currentSession >= totalSessions}
                        className="text-white hover:bg-white/20 h-8 w-8 p-0"
                      >
                        <ChevronRight size={16} />
                      </Button>
                    </div>
                  </div>
                  
                  {/* Course Progress */}
                  <div className="mb-3">
                    <div className="flex justify-between text-sm mb-2">
                      <span>Course Progress</span>
                      <span>{Math.round(progress)}%</span>
                    </div>
                    <Progress value={progress} className="bg-white/20" />
                  </div>
                  
                  {/* Current Session */}
                  <div className="mb-3">
                    <p className="text-white/90 text-sm">
                      <span className="font-medium">Currently: </span>
                      {sessions[currentSession - 1]?.title}
                    </p>
                  </div>
                  
                  <Button variant="ghost" size="sm" className="text-white hover:bg-white/20 border border-white/30 w-full">
                    {classType === "crib" ? "🏆 I Did It!" :
                     classType === "clicker" ? "✅ Mission Done" :
                     "✓ Mark Complete"}
                  </Button>
                </div>
                
                <CardContent className="p-4">
                  <div className="space-y-3">
                    <div className="flex items-center justify-between text-sm">
                      <div className="flex items-center gap-2">
                        <Clock size={16} className="text-gray-600" />
                        <span>Session Time</span>
                      </div>
                      <span className="font-medium">{this.formatTime(sessionTime)}</span>
                    </div>
                    <div className="flex items-center justify-between text-sm">
                      <div className="flex items-center gap-2">
                        <Clock size={16} className="text-gray-600" />
                        <span>Total Course Time</span>
                      </div>
                      <span className="font-medium">{this.formatTime(totalCourseTime)}</span>
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Session Navigation */}
              <Card>
                <CardHeader>
                  <h3>
                    {classType === "crib" ? "🌟 My Adventures" :
                     classType === "clicker" ? "🎯 Mission List" :
                     "📚 Module Index"}
                  </h3>
                </CardHeader>
                <CardContent>
                  <Tabs value={activeTab} onValueChange={(value) => this.setState({ activeTab: value })}>
                    <TabsList className="grid w-full grid-cols-3">
                      <TabsTrigger value="videos">
                        {classType === "crib" ? "🎬 Videos" :
                         classType === "clicker" ? "🎮 Videos" :
                         "📹 Videos"}
                      </TabsTrigger>
                      <TabsTrigger value="projects">
                        {classType === "crib" ? "🎨 My Art" :
                         classType === "clicker" ? "🚀 Projects" :
                         "💼 Portfolio"}
                      </TabsTrigger>
                      <TabsTrigger value="resources">
                        {classType === "crib" ? "📚 Fun Stuff" :
                         classType === "clicker" ? "🛠️ Toolkit" :
                         "📖 Resources"}
                      </TabsTrigger>
                    </TabsList>
                    
                    <TabsContent value="videos" className="space-y-2 mt-4">
                      {sessions.map((session, index) => (
                        <div
                          key={session.id}
                          onClick={() => !session.isLocked && onNavigateToSession(index + 1)}
                          className={`p-3 rounded-lg border cursor-pointer transition-colors ${
                            index + 1 === currentSession 
                              ? "border-light-blue bg-light-blue/10" 
                              : "border-gray-200 hover:bg-gray-50"
                          } ${session.isLocked ? "opacity-50 cursor-not-allowed" : ""}`}
                        >
                          <div className="flex items-center gap-3">
                            {session.isCompleted ? (
                              <CheckCircle className="text-green-500" size={16} />
                            ) : session.isLocked ? (
                              <Circle className="text-gray-400" size={16} />
                            ) : (
                              <Play className="text-light-blue" size={16} />
                            )}
                            <div className="flex-1">
                              <p className="text-sm">{session.title}</p>
                              <p className="text-xs text-gray-500">{session.duration}</p>
                            </div>
                            {index + 1 === currentSession && (
                              <Badge variant="secondary" className="text-xs">
                                Current
                              </Badge>
                            )}
                          </div>
                        </div>
                      ))}
                    </TabsContent>
                    
                    <TabsContent value="projects" className="space-y-4 mt-4">
                      <div className="space-y-4">
                        <h4 className="flex items-center gap-2">
                          <FileText size={16} />
                          {classType === "crib" ? "🎨 My Creation" :
                           classType === "clicker" ? "🚀 My Project" :
                           "📋 Assignment Submission"}
                        </h4>
                        
                        <Card className="p-4">
                          <h5 className="font-medium mb-2">
                            {classType === "crib" ? "🌟 Share Your Magic!" :
                             classType === "clicker" ? "🏆 Submit Your Quest!" :
                             "📤 Submit Your Work"}
                          </h5>
                          <p className="text-sm text-gray-600 mb-3">
                            {classType === "crib" ? `🎭 Show us the magical creation you made in Session ${currentSession}!` :
                             classType === "clicker" ? `⚡ Upload your awesome project from Mission ${currentSession}!` :
                             `📊 Submit your completed assignment for Module ${currentSession}`}
                          </p>
                          
                          <div className="space-y-3">
                            <div>
                              <Label htmlFor="assignment-upload">Choose File</Label>
                              <Input
                                id="assignment-upload"
                                type="file"
                                onChange={this.handleFileUpload}
                                accept=".pdf,.doc,.docx,.zip,.sb3"
                                className="mt-1"
                              />
                              <p className="text-xs text-gray-500 mt-1">
                                Accepted formats: PDF, DOC, ZIP, SB3
                              </p>
                            </div>
                            
                            {uploadedFile && (
                              <div className="flex items-center gap-2 p-2 bg-green-50 rounded border">
                                <CheckCircle className="text-green-500" size={16} />
                                <span className="text-sm text-green-700">
                                  {uploadedFile.name}
                                </span>
                              </div>
                            )}
                            
                            <Button 
                              size="sm" 
                              className="w-full"
                              disabled={!uploadedFile}
                            >
                              <Upload size={16} className="mr-2" />
                              {classType === "crib" ? "🎉 Share My Art!" :
                               classType === "clicker" ? "🚀 Submit Mission!" :
                               "📤 Submit Assignment"}
                            </Button>
                          </div>
                        </Card>
                        
                        <div className="space-y-2">
                          <h5 className="font-medium">
                            {classType === "crib" ? "🌈 My Art Gallery" :
                             classType === "clicker" ? "🏆 Completed Quests" :
                             "📋 Submission History"}
                          </h5>
                          <div className="space-y-2">
                            <div className="flex items-center justify-between p-2 bg-gray-50 rounded">
                              <div>
                                <p className="text-sm">Session 2 - Basic Concepts</p>
                                <p className="text-xs text-gray-500">Submitted 2 days ago</p>
                              </div>
                              <Badge variant="secondary" className="bg-green-100 text-green-700">
                                Graded
                              </Badge>
                            </div>
                            <div className="flex items-center justify-between p-2 bg-gray-50 rounded">
                              <div>
                                <p className="text-sm">Session 1 - Getting Started</p>
                                <p className="text-xs text-gray-500">Submitted 1 week ago</p>
                              </div>
                              <Badge variant="secondary" className="bg-green-100 text-green-700">
                                Graded
                              </Badge>
                            </div>
                          </div>
                        </div>
                      </div>
                    </TabsContent>
                    
                    <TabsContent value="resources" className="space-y-3 mt-4">
                      {resources.map((resource) => {
                        const Icon = resource.icon;
                        return (
                          <div
                            key={resource.id}
                            className="flex items-center justify-between p-3 border rounded-lg hover:bg-gray-50 cursor-pointer"
                            onClick={() => {
                              toast.success(`Opening ${resource.title}...`);
                            }}
                          >
                            <div className="flex items-center gap-3">
                              <Icon 
                                size={20} 
                                className={
                                  resource.type === "youtube" ? "text-red-500" :
                                  resource.type === "pdf" ? "text-red-600" :
                                  resource.type === "presentation" ? "text-orange-500" :
                                  "text-blue-500"
                                } 
                              />
                              <div>
                                <p className="text-sm font-medium">{resource.title}</p>
                                <p className="text-xs text-gray-500 capitalize">
                                  {resource.type === "youtube" ? "Video Tutorial" :
                                   resource.type === "pdf" ? "PDF Document" :
                                   resource.type === "presentation" ? "Google Slides" :
                                   "Google Docs"}
                                </p>
                              </div>
                            </div>
                            <ExternalLink size={16} className="text-gray-400" />
                          </div>
                        );
                      })}
                    </TabsContent>
                  </Tabs>
                </CardContent>
              </Card>

              {/* Help Section */}
              <Card>
                <CardHeader>
                  <h3 className="flex items-center gap-2">
                    <HelpCircle className="text-light-blue" size={20} />
                    Get Help
                  </h3>
                </CardHeader>
                <CardContent>
                  <p className="text-sm text-gray-600 mb-3">
                    Need assistance with this lesson?
                  </p>
                  <Button 
                    variant="outline" 
                    className="w-full"
                    onClick={() => this.setState({ isSupportModalOpen: true })}
                  >
                    Contact Support
                  </Button>
                </CardContent>
              </Card>
            </div>
          </div>
        </div>
        
        {/* Support Modal */}
        <SupportModal 
          isOpen={isSupportModalOpen}
          onClose={() => this.setState({ isSupportModalOpen: false })}
          onSelectMentor={onSelectMentor}
          classType={classType}
        />
      </div>
    );
  }
}