import { Card, CardContent, CardHeader } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Progress } from "./ui/progress";
import { 
  BookOpen, 
  Clock, 
  Trophy, 
  Target,
  TrendingUp,
  Calendar,
  Play
} from "lucide-react";

export function DashboardView({ onNavigateToClasses }) {
  const recentActivity = [
    {
      course: "Magic Code Blocks Adventure",
      lesson: "Session 3: Fun Commands",
      progress: 25,
      classType: "crib"
    },
    {
      course: "Game Builder Academy", 
      lesson: "Session 7: Loops and Conditions",
      progress: 47,
      classType: "clicker"
    },
    {
      course: "Python Programming Mastery",
      lesson: "Session 5: Functions",
      progress: 31,
      classType: "quest"
    }
  ];

  // Age-appropriate achievements
  const getAchievements = (classType) => {
    switch (classType) {
      case "crib":
        return [
          { title: "🌟 First Magic!", description: "Completed your first magical lesson", earned: true },
          { title: "🌈 Rainbow Learner", description: "Studied 5 days in a row", earned: true },
          { title: "🎮 Block Builder", description: "Built 10 colorful block creations", earned: false },
          { title: "🏆 Little Wizard", description: "Solved 25 fun puzzles", earned: false },
        ];
      case "clicker":
        return [
          { title: "🚀 Code Rookie", description: "Completed your first coding mission", earned: true },
          { title: "⚡ Power Coder", description: "Coded 5 days in a row", earned: true },
          { title: "🎯 Game Master", description: "Built 10 awesome games", earned: false },
          { title: "🏆 Tech Hero", description: "Conquered 25 coding challenges", earned: false },
        ];
      case "quest":
        return [
          { title: "Developer Genesis", description: "Completed your first programming module", earned: true },
          { title: "Consistency Pro", description: "Maintained a 5-day coding streak", earned: true },
          { title: "Project Builder", description: "Deployed 10 functional applications", earned: false },
          { title: "Problem Solver", description: "Resolved 25 complex algorithms", earned: false },
        ];
      default:
        return [
          { title: "First Steps", description: "Completed your first lesson", earned: true },
          { title: "Week Warrior", description: "Studied 5 days in a row", earned: true },
          { title: "Code Explorer", description: "Completed 10 coding exercises", earned: false },
          { title: "Problem Solver", description: "Solved 25 challenges", earned: false },
        ];
    }
  };

  // Age-appropriate tasks
  const getUpcomingTasks = (classType) => {
    switch (classType) {
      case "crib":
        return [
          { title: "🎨 Finish Rainbow Art Project", course: "Magic Code Blocks", dueDate: "Today" },
          { title: "🎭 Create Fun Animation", course: "Digital Art Studio", dueDate: "Tomorrow" },
          { title: "🌟 Show Your Creation", course: "Animation Fun", dueDate: "Dec 28" },
        ];
      case "clicker":
        return [
          { title: "🎮 Complete Game Level 4", course: "Game Builder Academy", dueDate: "Today" },
          { title: "🌐 Build Your Website", course: "Website Wizard", dueDate: "Tomorrow" },
          { title: "🤖 Program Your Robot", course: "Robot Adventures", dueDate: "Dec 28" },
        ];
      case "quest":
        return [
          { title: "Deploy Portfolio Project", course: "Python Mastery", dueDate: "Today" },
          { title: "Submit Full Stack Application", course: "Development Bootcamp", dueDate: "Tomorrow" },
          { title: "Complete AI Model Training", course: "Machine Learning", dueDate: "Dec 28" },
        ];
      default:
        return [
          { title: "Complete Session 4", course: "Pictoblox Basics", dueDate: "Today" },
          { title: "Submit Assignment 2", course: "Advanced Programming", dueDate: "Tomorrow" },
          { title: "Review Session 6", course: "Python Fundamentals", dueDate: "Dec 28" },
        ];
    }
  };

  // Determine primary class type based on recent activity
  const primaryClassType = recentActivity[0]?.classType || "clicker";
  const achievements = getAchievements(primaryClassType);
  const upcomingTasks = getUpcomingTasks(primaryClassType);

  return (
    <div className="p-6 space-y-6">
      {/* Welcome Section - Age Appropriate */}
      <div className={`text-white p-6 rounded-lg ${
        primaryClassType === 'crib' ? 'gradient-crib-welcome' :
        primaryClassType === 'clicker' ? 'gradient-clicker-welcome' :
        'gradient-quest-welcome'
      }`}>
        <h1 className="text-2xl mb-2">
          {primaryClassType === 'crib' ? '🌟 Welcome back, Little Coder!' :
           primaryClassType === 'clicker' ? '⚡ Hey there, Tech Explorer!' :
           'Welcome back, Developer!'}
        </h1>
        <p className="text-white/90 mb-4">
          {primaryClassType === 'crib' ? 'Ready for more magical coding adventures?' :
           primaryClassType === 'clicker' ? 'Ready to build something awesome today?' :
           'Ready to level up your programming skills?'}
        </p>
        <Button 
          onClick={onNavigateToClasses}
          className={
            primaryClassType === 'crib' 
              ? 'bg-white hover:bg-white/90 text-navy' 
              : 'bg-yellow hover:bg-yellow/90 text-navy'
          }
        >
          <Play size={16} className="mr-2" />
          {primaryClassType === 'crib' ? '🎮 Continue Playing' :
           primaryClassType === 'clicker' ? '🚀 Continue Building' :
           'Continue Development'}
        </Button>
      </div>

      {/* Stats Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 bg-light-blue/10 rounded-lg">
                <BookOpen className="text-light-blue" size={20} />
              </div>
              <div>
                <p className="text-2xl">3</p>
                <p className="text-sm text-gray-600">
                  {primaryClassType === 'crib' ? '🎨 Fun Adventures' :
                   primaryClassType === 'clicker' ? '🎯 Active Quests' :
                   'Active Programs'}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 bg-yellow/10 rounded-lg">
                <Clock className="text-yellow" size={20} />
              </div>
              <div>
                <p className="text-2xl">24h</p>
                <p className="text-sm text-gray-600">
                  {primaryClassType === 'crib' ? '⏰ Play Time' :
                   primaryClassType === 'clicker' ? '🕐 Build Time' :
                   'Development Hours'}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 bg-navy/10 rounded-lg">
                <Trophy className="text-navy" size={20} />
              </div>
              <div>
                <p className="text-2xl">15</p>
                <p className="text-sm text-gray-600">
                  {primaryClassType === 'crib' ? '🌟 Stories Finished' :
                   primaryClassType === 'clicker' ? '🏆 Missions Complete' :
                   'Modules Completed'}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 bg-green-500/10 rounded-lg">
                <Target className="text-green-500" size={20} />
              </div>
              <div>
                <p className="text-2xl">85%</p>
                <p className="text-sm text-gray-600">
                  {primaryClassType === 'crib' ? '⭐ Magic Stars' :
                   primaryClassType === 'clicker' ? '🎯 Success Rate' :
                   'Performance Score'}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Recent Activity */}
        <Card>
          <CardHeader>
            <h3 className="flex items-center gap-2">
              <TrendingUp className="text-light-blue" size={20} />
              {primaryClassType === 'crib' ? '🌈 My Adventures' :
               primaryClassType === 'clicker' ? '⚡ Recent Builds' :
               'Recent Progress'}
            </h3>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {recentActivity.map((activity, index) => (
                <div key={index} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                  <div className="flex-1">
                    <h4 className="text-sm">{activity.course}</h4>
                    <p className="text-xs text-gray-500 mb-2">{activity.lesson}</p>
                    <Progress value={activity.progress} className="h-2" />
                  </div>
                  <Badge 
                    variant="outline" 
                    className={`ml-3 ${
                      activity.classType === 'crib' ? 'text-light-blue border-light-blue' :
                      activity.classType === 'clicker' ? 'text-navy border-navy' :
                      'text-navy border-navy'
                    }`}
                  >
                    {activity.classType}
                  </Badge>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        {/* Upcoming Tasks */}
        <Card>
          <CardHeader>
            <h3 className="flex items-center gap-2">
              <Calendar className="text-light-blue" size={20} />
              {primaryClassType === 'crib' ? '🎯 Fun To-Do' :
               primaryClassType === 'clicker' ? '🚀 Next Missions' :
               'Upcoming Milestones'}
            </h3>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {upcomingTasks.map((task, index) => (
                <div key={index} className="flex items-center justify-between p-3 border rounded-lg">
                  <div>
                    <h4 className="text-sm">{task.title}</h4>
                    <p className="text-xs text-gray-500">{task.course}</p>
                  </div>
                  <Badge variant="outline" className="text-xs">
                    {task.dueDate}
                  </Badge>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Achievements */}
      <Card>
        <CardHeader>
          <h3 className="flex items-center gap-2">
            <Trophy className="text-yellow" size={20} />
            {primaryClassType === 'crib' ? '🏆 My Trophies' :
             primaryClassType === 'clicker' ? '🎖️ My Badges' :
             'Professional Achievements'}
          </h3>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
            {achievements.map((achievement, index) => (
              <div 
                key={index} 
                className={`p-4 border rounded-lg text-center ${
                  achievement.earned ? 'bg-yellow/10 border-yellow' : 'bg-gray-50 border-gray-200'
                }`}
              >
                <Trophy 
                  className={`mx-auto mb-2 ${
                    achievement.earned ? 'text-yellow' : 'text-gray-400'
                  }`} 
                  size={24} 
                />
                <h4 className="text-sm mb-1">{achievement.title}</h4>
                <p className="text-xs text-gray-600">{achievement.description}</p>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>
    </div>
  );
}