import { Card, CardContent, CardHeader } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Progress } from "./ui/progress";
import { Play, BookOpen, Clock, Star } from "lucide-react";
import { ClassNavbar } from "./ClassNavbar";
import { ImageWithFallback } from "./figma/ImageWithFallback";

export function CourseOverview({
  classType,
  courses,
  onBackToClasses,
  onViewCourse,
  onPayForCourse
}) {
  const getClassConfig = () => {
    switch (classType) {
      case "crib":
        return {
          className: "Crib Adventures",
          description: "Colorful adventures in coding for young minds",
          gradient: "gradient-crib-course",
          accent: "text-yellow",
          defaultImage: "https://images.unsplash.com/photo-1565229284535-2cbbe3049123?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxjb2RpbmclMjBibG9ja3MlMjBjb2xvcmZ1bCUyMGtpZHMlMjBwcm9ncmFtbWluZ3xlbnwxfHx8fDE3NTg4ODE0MzF8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral"
        };
      case "clicker":
        return {
          className: "Clicker Challenges",
          description: "Engaging coding challenges for intermediate learners",
          gradient: "gradient-clicker-course",
          accent: "text-light-blue",
          defaultImage: "https://images.unsplash.com/photo-1717588282722-ab1beb899c26?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxnYW1lJTIwZGV2ZWxvcG1lbnQlMjBraWRzJTIwdGVjaG5vbG9neXxlbnwxfHx8fDE3NTg4ODE0MzR8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral"
        };
      case "quest":
        return {
          className: "Quest Mastery",
          description: "Advanced programming for teens and beyond",
          gradient: "gradient-quest-course",
          accent: "text-navy",
          defaultImage: "https://images.unsplash.com/photo-1573496528681-9b0f4fb0c660?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxweXRob24lMjBwcm9ncmFtbWluZyUyMGFkdmFuY2VkJTIwY29kaW5nfGVufDF8fHx8MTc1ODg4MTQzN3ww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral"
        };
    }
  };

  const config = getClassConfig();
  const paidCourses = courses.filter(course => course.isPaid);
  const unpaidCourses = courses.filter(course => !course.isPaid);

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Class Navbar */}
      <ClassNavbar 
        classType={classType}
        onBackToClasses={onBackToClasses}
      />

      {/* Class Header */}
      <div className={`${config.gradient} text-white p-8`}>
        <div className="max-w-7xl mx-auto">
          <h1 className="text-4xl font-bold mb-3">{config.className}</h1>
          <p className="text-white/90 text-lg">{config.description}</p>
          
          <div className="flex items-center gap-6 mt-6">
            <div className="flex items-center gap-2">
              <BookOpen className="w-5 h-5" />
              <span>{courses.length} Courses</span>
            </div>
            <div className="flex items-center gap-2">
              <Play className="w-5 h-5" />
              <span>{paidCourses.length} Enrolled</span>
            </div>
            <div className="flex items-center gap-2">
              <Star className="w-5 h-5" />
              <span>Expert Support</span>
            </div>
          </div>
        </div>
      </div>

      {/* Course Content */}
      <div className="max-w-7xl mx-auto p-8">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Main Content */}
          <div className="lg:col-span-2 space-y-8">
            {/* Enrolled Courses */}
            {paidCourses.length > 0 && (
              <div>
                <h2 className="text-2xl font-semibold mb-6">Continue Learning</h2>
                <div className="space-y-4">
                  {paidCourses.map((course) => (
                    <Card key={course.id} className="overflow-hidden">
                      <CardContent className="p-0">
                        <div className="flex">
                          <div className="w-48 h-32">
                            <ImageWithFallback
                              src={config.defaultImage}
                              alt={course.title}
                              className="w-full h-full object-cover"
                            />
                          </div>
                          <div className="flex-1 p-6">
                            <h3 className="text-xl font-semibold mb-2">{course.title}</h3>
                            <div className="flex items-center gap-4 text-sm text-gray-600 mb-4">
                              <div className="flex items-center gap-1">
                                <BookOpen className="w-4 h-4" />
                                <span>{course.sessions} Sessions</span>
                              </div>
                              <div className="flex items-center gap-1">
                                <Clock className="w-4 h-4" />
                                <span>Session {course.currentSession}/{course.sessions}</span>
                              </div>
                            </div>
                            
                            <div className="mb-4">
                              <div className="flex justify-between text-sm mb-1">
                                <span>Progress</span>
                                <span>{Math.round((course.currentSession / course.sessions) * 100)}%</span>
                              </div>
                              <Progress value={(course.currentSession / course.sessions) * 100} />
                            </div>
                            
                            <Button 
                              onClick={() => onViewCourse(course.id)}
                              className="bg-light-blue hover:bg-light-blue/90 text-white"
                            >
                              <Play className="w-4 h-4 mr-2" />
                              Continue Course
                            </Button>
                          </div>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              </div>
            )}

            {/* Available Courses */}
            {unpaidCourses.length > 0 && (
              <div>
                <h2 className="text-2xl font-semibold mb-6">Available Courses</h2>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  {unpaidCourses.map((course) => (
                    <Card key={course.id} className="overflow-hidden">
                      <div className="h-40">
                        <ImageWithFallback
                          src={config.defaultImage}
                          alt={course.title}
                          className="w-full h-full object-cover"
                        />
                      </div>
                      <CardContent className="p-4">
                        <h3 className="font-semibold mb-2">{course.title}</h3>
                        <div className="flex items-center gap-4 text-sm text-gray-600 mb-4">
                          <div className="flex items-center gap-1">
                            <BookOpen className="w-4 h-4" />
                            <span>{course.sessions} Sessions</span>
                          </div>
                          <Badge variant="outline">${course.price}</Badge>
                        </div>
                        
                        <Button 
                          variant="outline"
                          className="w-full"
                          onClick={() => onPayForCourse(course.id)}
                        >
                          Unlock Course
                        </Button>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              </div>
            )}
          </div>

          {/* Sidebar */}
          <div className="space-y-6">
            <Card>
              <CardHeader>
                <h3>Class Overview</h3>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  <div className="flex justify-between">
                    <span className="text-gray-600">Total Courses</span>
                    <span className="font-medium">{courses.length}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-gray-600">Enrolled</span>
                    <span className="font-medium">{paidCourses.length}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-gray-600">Completed</span>
                    <span className="font-medium">{courses.filter(c => c.isCompleted).length}</span>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <h3>Class Features</h3>
              </CardHeader>
              <CardContent>
                <div className="space-y-3 text-sm">
                  <div className="flex items-center gap-2">
                    <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                    <span>Interactive Video Lessons</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                    <span>Hands-on Projects</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                    <span>Expert Mentor Support</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                    <span>Certificate of Completion</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                    <span>PictoBlox Integration</span>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </div>
  );
}
