import { ClassCard } from "./ClassCard";
import { Card, CardContent, CardHeader } from "./ui/card";
import { Trophy, Clock, Target, Star, TrendingUp, Calendar } from "lucide-react";

interface ClassesViewProps {
  onViewCourse: (courseId: string, classType: string) => void;
  onPayForCourse: (courseId: string) => void;
  selectedClass: "crib" | "clicker" | "quest";
}

export function ClassesView({ onViewCourse, onPayForCourse, selectedClass }: ClassesViewProps) {
  const mockClasses = [
    {
      classType: "crib" as const,
      title: "Crib",
      description: "🌟 Magical coding adventures! Learn through colorful stories, friendly characters, and super fun games made just for little learners like you!",
      courses: [
        {
          id: "crib-1",
          title: "Magic Code Blocks Adventure",
          sessions: 12,
          currentSession: 3,
          isPaid: true,
          isCompleted: false,
        },
        {
          id: "crib-2", 
          title: "Rainbow Digital Art Studio",
          sessions: 10,
          currentSession: 1,
          isPaid: false,
          isCompleted: false,
          price: 29.99,
        },
        {
          id: "crib-3",
          title: "Fun Animations for Beginners",
          sessions: 8,
          currentSession: 1,
          isPaid: false,
          isCompleted: false,
          price: 24.99,
        }
      ]
    },
    {
      classType: "clicker" as const,
      title: "Clicker",
      description: "⚡ Epic coding quests! Create amazing games, build awesome websites, and become a tech hero with super cool hands-on projects!",
      courses: [
        {
          id: "clicker-1",
          title: "Game Builder Academy",
          sessions: 15,
          currentSession: 7,
          isPaid: true,
          isCompleted: false,
        },
        {
          id: "clicker-2",
          title: "Website Wizard Course",
          sessions: 12,
          currentSession: 1,
          isPaid: false,
          isCompleted: false,
          price: 39.99,
        },
        {
          id: "clicker-3",
          title: "Robot Programming Adventures",
          sessions: 10,
          currentSession: 1,
          isPaid: false,
          isCompleted: false,
          price: 49.99,
        }
      ]
    },
    {
      classType: "quest" as const,
      title: "Quest",
      description: "🚀 Professional development pathway. Master advanced programming concepts and build industry-standard applications for your portfolio.",
      courses: [
        {
          id: "quest-1",
          title: "Python Programming Mastery",
          sessions: 16,
          currentSession: 5,
          isPaid: true,
          isCompleted: false,
        },
        {
          id: "quest-2",
          title: "Full Stack Development Bootcamp",
          sessions: 14,
          currentSession: 1,
          isPaid: false,
          isCompleted: false,
          price: 59.99,
        },
        {
          id: "quest-3",
          title: "Advanced AI & Machine Learning",
          sessions: 12,
          currentSession: 1,
          isPaid: false,
          isCompleted: false,
          price: 69.99,
        }
      ]
    }
  ];

  // Filter classes to only show the selected class
  const filteredClasses = mockClasses.filter((classData) => classData.classType === selectedClass);
  const currentClass = filteredClasses[0]; // There should only be one matching class

  // Get the class-specific title based on selected class
  const getClassTitle = () => {
    switch (selectedClass) {
      case "crib":
        return "🧸 Crib Adventures - My Courses";
      case "clicker":
        return "🎮 Clicker Academy - My Courses";
      case "quest":
        return "🚀 Quest Mastery - My Courses";
      default:
        return "🎓 My Learning Journey";
    }
  };

  // Calculate stats
  const totalCourses = currentClass?.courses.length || 0;
  const activeCourses = currentClass?.courses.filter(c => c.isPaid && !c.isCompleted).length || 0;
  const totalSessions = currentClass?.courses.reduce((acc, course) => {
    if (course.isPaid) return acc + course.currentSession;
    return acc;
  }, 0) || 0;

  // Get class-specific content
  const getClassContent = () => {
    switch (selectedClass) {
      case "crib":
        return {
          statsTitle: "🌟 My Learning Stars",
          tipTitle: "💡 Fun Learning Tip",
          tip: "Take breaks every 20 minutes! Stand up, stretch, and wiggle around. Your brain learns better when you move!",
          upcomingTitle: "📅 Next Adventures",
          streakEmoji: "🔥",
          progressLabel: "Adventures in Progress"
        };
      case "clicker":
        return {
          statsTitle: "📊 My Tech Stats",
          tipTitle: "💡 Pro Tip",
          tip: "Challenge yourself to build something new every week! The best way to level up your coding skills is by creating projects.",
          upcomingTitle: "📅 Upcoming Missions",
          streakEmoji: "⚡",
          progressLabel: "Active Challenges"
        };
      case "quest":
        return {
          statsTitle: "📊 Learning Analytics",
          tipTitle: "💡 Professional Insight",
          tip: "Build a portfolio as you learn. Document your projects on GitHub and create case studies to showcase your skills to future employers.",
          upcomingTitle: "📅 Upcoming Sessions",
          streakEmoji: "🎯",
          progressLabel: "Courses in Progress"
        };
      default:
        return {
          statsTitle: "📊 My Stats",
          tipTitle: "💡 Tip",
          tip: "Stay consistent with your learning journey!",
          upcomingTitle: "📅 Upcoming",
          streakEmoji: "🔥",
          progressLabel: "In Progress"
        };
    }
  };

  const content = getClassContent();

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="px-6 py-8">
        <div className="max-w-7xl mx-auto">
          <div className="space-y-6">
            {/* Quick Stats Overview */}
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
              <Card className="border-2 border-light-blue/20 hover:border-light-blue/40 transition-colors">
                <CardContent className="p-6">
                  <div className="flex items-center gap-3">
                    <div className={`w-12 h-12 rounded-lg flex items-center justify-center ${
                      selectedClass === "crib" ? "gradient-crib-avatar" :
                      selectedClass === "clicker" ? "gradient-clicker-avatar" :
                      "gradient-quest-avatar"
                    }`}>
                      <Trophy className="w-6 h-6 text-white" />
                    </div>
                    <div>
                      <p className="text-sm text-gray-600">{content.progressLabel}</p>
                      <p className="text-2xl text-navy">{activeCourses}</p>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card className="border-2 border-yellow/20 hover:border-yellow/40 transition-colors">
                <CardContent className="p-6">
                  <div className="flex items-center gap-3">
                    <div className="w-12 h-12 bg-yellow/20 rounded-lg flex items-center justify-center">
                      <Target className="w-6 h-6 text-yellow" />
                    </div>
                    <div>
                      <p className="text-sm text-gray-600">Total Courses</p>
                      <p className="text-2xl text-navy">{totalCourses}</p>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card className="border-2 border-navy/20 hover:border-navy/40 transition-colors">
                <CardContent className="p-6">
                  <div className="flex items-center gap-3">
                    <div className="w-12 h-12 bg-navy/10 rounded-lg flex items-center justify-center">
                      <Clock className="w-6 h-6 text-navy" />
                    </div>
                    <div>
                      <p className="text-sm text-gray-600">Sessions Done</p>
                      <p className="text-2xl text-navy">{totalSessions}</p>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card className="border-2 border-light-blue/20 hover:border-light-blue/40 transition-colors">
                <CardContent className="p-6">
                  <div className="flex items-center gap-3">
                    <div className="w-12 h-12 bg-gradient-to-br from-yellow to-orange-400 rounded-lg flex items-center justify-center">
                      <Star className="w-6 h-6 text-white" />
                    </div>
                    <div>
                      <p className="text-sm text-gray-600">Study Streak</p>
                      <p className="text-2xl text-navy">5 {content.streakEmoji}</p>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Main Course Card */}
            {currentClass && (
              <div className="w-full">
                <ClassCard
                  classType={currentClass.classType}
                  title={currentClass.title}
                  description={currentClass.description}
                  courses={currentClass.courses}
                  onViewCourse={(courseId) => onViewCourse(courseId, currentClass.classType)}
                  onPayForCourse={onPayForCourse}
                />
              </div>
            )}

            {/* Additional Info Cards */}
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {/* Learning Tip */}
              <Card className="border-2 border-yellow/20">
                <CardHeader className="pb-3">
                  <h3 className="text-lg text-navy">{content.tipTitle}</h3>
                </CardHeader>
                <CardContent>
                  <p className="text-gray-700">{content.tip}</p>
                </CardContent>
              </Card>

              {/* Progress Tracker */}
              <Card className="border-2 border-light-blue/20">
                <CardHeader className="pb-3">
                  <h3 className="text-lg text-navy">
                    <TrendingUp className="w-5 h-5 inline mr-2 text-light-blue" />
                    Your Progress This Week
                  </h3>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-gray-600">
                        {selectedClass === "crib" ? "Lessons Completed" :
                         selectedClass === "clicker" ? "Challenges Solved" :
                         "Modules Finished"}
                      </span>
                      <span className="text-light-blue">12</span>
                    </div>
                    <div className="w-full bg-gray-200 rounded-full h-2">
                      <div className="gradient-clicker-card h-2 rounded-full" style={{ width: '75%' }}></div>
                    </div>
                    <p className="text-xs text-gray-500">Keep it up! You're doing great! 🎉</p>
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Upcoming Sessions Preview */}
            <Card className="border-2 border-navy/20">
              <CardHeader className="pb-3">
                <h3 className="text-lg text-navy">
                  <Calendar className="w-5 h-5 inline mr-2 text-navy" />
                  {content.upcomingTitle}
                </h3>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  {currentClass?.courses.filter(c => c.isPaid && !c.isCompleted).slice(0, 2).map((course, index) => (
                    <div key={course.id} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg border border-gray-200">
                      <div className="flex items-center gap-3">
                        <div className={`w-10 h-10 rounded-lg flex items-center justify-center ${
                          selectedClass === "crib" ? "gradient-crib-avatar" :
                          selectedClass === "clicker" ? "gradient-clicker-avatar" :
                          "gradient-quest-avatar"
                        }`}>
                          <span className="text-white">{course.currentSession}</span>
                        </div>
                        <div>
                          <p className="text-navy">{course.title}</p>
                          <p className="text-xs text-gray-500">Session {course.currentSession} of {course.sessions}</p>
                        </div>
                      </div>
                      <div className="text-xs text-gray-500">
                        {index === 0 ? "Continue" : "Next up"}
                      </div>
                    </div>
                  ))}
                  {(!currentClass?.courses.some(c => c.isPaid && !c.isCompleted)) && (
                    <div className="text-center py-6 text-gray-500">
                      <p>
                        {selectedClass === "crib" ? "🎉 Pick a new adventure to start!" :
                         selectedClass === "clicker" ? "🚀 Ready for your next challenge!" :
                         "📚 Select a course to begin your learning journey!"}
                      </p>
                    </div>
                  )}
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </div>
  );
}