import { Card, CardContent } from "./ui/card";
import { Button } from "./ui/button";
import { Heart, Zap, Target, ChevronRight } from "lucide-react";
import companyLogo from "figma:asset/517ee9d784589cb130eb229f0ac854531acfb4ba.png";

export function ClassSelectionView({ onSelectClass }) {
  const classOptions = [
    {
      type: "crib",
      title: "Crib Adventures",
      subtitle: "Ages 5-7 years",
      description: "🌟 Magical learning through play and creativity! Start your coding journey with colorful blocks, fun animations, and exciting adventures.",
      features: [
        "🎨 Visual drag-and-drop coding",
        "🎭 Story-based learning",
        "🌈 Colorful and engaging interface",
        "⭐ Achievement badges and rewards"
      ],
      icon: <Heart className="w-12 h-12 text-yellow" fill="currentColor" />,
      gradient: "from-light-blue via-light-blue/80 to-yellow/30",
      buttonColor: "bg-yellow hover:bg-yellow/90 text-navy",
      borderColor: "border-yellow/40"
    },
    {
      type: "clicker",
      title: "Clicker Academy",
      subtitle: "Ages 8-12 years",
      description: "🚀 Power up your tech skills! Build games, create websites, and master programming through exciting missions and challenges.",
      features: [
        "🎮 Game development projects",
        "💻 Interactive coding challenges",
        "🔧 Real programming tools",
        "🏆 Level-based progression"
      ],
      icon: <Zap className="w-12 h-12 text-light-blue" fill="currentColor" />,
      gradient: "from-navy via-navy/85 to-light-blue/60",
      buttonColor: "bg-light-blue hover:bg-light-blue/90 text-white",
      borderColor: "border-light-blue/50"
    },
    {
      type: "quest",
      title: "Quest Mastery",
      subtitle: "Ages 13+ years",
      description: "💼 Professional-level programming mastery! Dive deep into advanced concepts, build real-world projects, and prepare for your tech career.",
      features: [
        "🏗️ Full-stack development",
        "🔬 Advanced algorithms & AI",
        "📊 Industry-standard tools",
        "🎯 Career preparation focus"
      ],
      icon: <Target className="w-12 h-12 text-light-blue" fill="currentColor" />,
      gradient: "from-navy via-navy/95 to-light-blue/25",
      buttonColor: "bg-light-blue hover:bg-light-blue/90 text-white",
      borderColor: "border-light-blue/35"
    }
  ];

  return (
    <div className="min-h-screen bg-gray-50 flex flex-col">
      {/* Header */}
      <div className="bg-white border-b border-gray-200 py-4 px-6">
        <div className="max-w-7xl mx-auto flex items-center justify-center">
          <img 
            src={companyLogo} 
            alt="DigiFunzi" 
            className="h-8 w-auto"
          />
          <h1 className="ml-3 text-2xl text-navy">DigiFunzi Learning Platform</h1>
        </div>
      </div>

      {/* Main Content */}
      <div className="flex-1 flex items-center justify-center p-6">
        <div className="w-full max-w-6xl">
          {/* Welcome Section */}
          <div className="text-center mb-12">
            <h2 className="text-4xl text-navy mb-4">Welcome to Your Learning Journey! 🌟</h2>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto">
              Choose your age-appropriate learning path to start building amazing projects and mastering new skills
            </p>
          </div>

          {/* Class Selection Cards */}
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
            {classOptions.map((classOption) => (
              <Card 
                key={classOption.type}
                className={`group hover:shadow-xl transition-all duration-300 cursor-pointer border-2 ${classOption.borderColor} hover:scale-105 overflow-hidden`}
                onClick={() => onSelectClass(classOption.type)}
              >
                {/* Card Header with Gradient */}
                <div className={`bg-gradient-to-r ${classOption.gradient} text-white p-6 text-center`}>
                  <div className="flex justify-center mb-4">
                    {classOption.icon}
                  </div>
                  <h3 className="text-2xl mb-2">{classOption.title}</h3>
                  <p className="text-white/90">{classOption.subtitle}</p>
                </div>

                <CardContent className="p-6">
                  {/* Description */}
                  <p className="text-gray-600 mb-6 leading-relaxed">
                    {classOption.description}
                  </p>

                  {/* Features List */}
                  <div className="space-y-3 mb-6">
                    {classOption.features.map((feature, index) => (
                      <div key={index} className="flex items-center gap-3">
                        <div className="w-6 h-6 rounded-full bg-gray-100 flex items-center justify-center flex-shrink-0">
                          <div className="w-2 h-2 rounded-full bg-light-blue"></div>
                        </div>
                        <span className="text-sm text-gray-700">{feature}</span>
                      </div>
                    ))}
                  </div>

                  {/* Action Button */}
                  <Button 
                    className={`w-full ${classOption.buttonColor} flex items-center justify-center gap-2 group-hover:scale-105 transition-transform`}
                    size="lg"
                  >
                    Start Learning Journey
                    <ChevronRight className="w-5 h-5" />
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>

          {/* Additional Info */}
          <div className="text-center mt-12">
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 max-w-4xl mx-auto">
              <h3 className="text-xl text-navy mb-4">✨ What Makes DigiFunzi Special?</h3>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-6 text-center">
                <div>
                  <div className="w-12 h-12 bg-light-blue/10 rounded-full flex items-center justify-center mx-auto mb-3">
                    <span className="text-2xl">🎯</span>
                  </div>
                  <h4 className="text-navy mb-2">Age-Appropriate Learning</h4>
                  <p className="text-sm text-gray-600">Carefully designed curriculum that matches your developmental stage</p>
                </div>
                <div>
                  <div className="w-12 h-12 bg-navy/10 rounded-full flex items-center justify-center mx-auto mb-3">
                    <span className="text-2xl">🚀</span>
                  </div>
                  <h4 className="text-navy mb-2">Hands-On Projects</h4>
                  <p className="text-sm text-gray-600">Build real projects while learning programming fundamentals</p>
                </div>
                <div>
                  <div className="w-12 h-12 bg-yellow/10 rounded-full flex items-center justify-center mx-auto mb-3">
                    <span className="text-2xl">👨‍🏫</span>
                  </div>
                  <h4 className="text-navy mb-2">Expert Mentors</h4>
                  <p className="text-sm text-gray-600">Get help from experienced mentors when you need it</p>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
