import { ArrowLeft, Home, Heart, Lightbulb, Shield, HelpCircle, MessageCircle } from "lucide-react";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";

interface ClassNavbarProps {
  classType: "crib" | "clicker" | "quest";
  currentCourse?: string;
  onBackToClasses?: () => void;
  onNavigateToDashboard?: () => void;
  onHelp?: () => void;
  onContactUs?: () => void;
}

export function ClassNavbar({ 
  classType, 
  currentCourse, 
  onBackToClasses,
  onNavigateToDashboard,
  onHelp,
  onContactUs 
}: ClassNavbarProps) {
  const getClassConfig = () => {
    switch (classType) {
      case "crib":
        return {
          gradient: "gradient-crib-navbar",
          textColor: "text-white",
          icon: <Heart className="w-5 h-5 text-navy" fill="currentColor" />,
          className: "Crib Class",
          ageRange: "Ages 5-7",
          badgeColor: "bg-light-blue text-white"
        };
      case "clicker":
        return {
          gradient: "gradient-clicker-navbar",
          textColor: "text-white",
          icon: <Lightbulb className="w-5 h-5 text-yellow" fill="currentColor" />,
          className: "Clicker Class",
          ageRange: "Ages 8-12",
          badgeColor: "bg-yellow text-navy"
        };
      case "quest":
        return {
          gradient: "gradient-quest-navbar",
          textColor: "text-white",
          icon: <Shield className="w-5 h-5 text-yellow" fill="currentColor" />,
          className: "Quest Class",
          ageRange: "Ages 13+",
          badgeColor: "bg-yellow text-white"
        };
    }
  };

  const config = getClassConfig();

  return (
    <div className={`${config.gradient} ${config.textColor}`}>
      {/* Top Navigation Bar */}
      <div className="flex items-center justify-between px-6 py-3 border-b border-white/20">
        <div className="flex items-center gap-4">
          <div className="flex items-center gap-2">
            <span className="text-lg font-bold">digifunzi</span>
            <div className="flex items-center gap-1">
              {config.icon}
              <span className="font-medium">{config.className}</span>
            </div>
          </div>
        </div>

        <div className="flex items-center gap-2">
          <Badge className={config.badgeColor}>
            Level 3
          </Badge>
          <Badge variant="outline" className="bg-white/10 border-white/30 text-white">
            180 XP
          </Badge>
          <div className="flex items-center gap-1">
            <div className="w-8 h-8 bg-green-500 rounded-full flex items-center justify-center text-white font-medium text-sm">
              A
            </div>
            <div className="w-8 h-8 bg-yellow rounded-full flex items-center justify-center text-white font-medium text-sm">
              ♠
            </div>
            <div className="w-8 h-8 bg-orange-500 rounded-full flex items-center justify-center text-white font-medium text-sm">
              S
            </div>
          </div>
          
          <Button
            variant="ghost"
            size="sm"
            onClick={onHelp}
            className={`${config.textColor} hover:bg-white/10 border border-white/30`}
          >
            <HelpCircle className="w-4 h-4 mr-1" />
            Help
          </Button>
          
          <Button
            variant="ghost"
            size="sm"
            onClick={onContactUs}
            className={`${config.textColor} hover:bg-white/10 border border-white/30`}
          >
            <MessageCircle className="w-4 h-4 mr-1" />
            Contact Us
          </Button>
        </div>
      </div>

      {/* Breadcrumb Navigation with white background */}
      <div className="bg-white border-b border-gray-200 px-6 py-3">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-2 text-sm text-gray-600">
            <button 
              onClick={onBackToClasses}
              className="flex items-center gap-1 text-light-blue hover:text-light-blue/80 transition-colors"
            >
              <ArrowLeft className="w-4 h-4" />
              <span>Back to Classes</span>
            </button>
            <span className="text-gray-400">|</span>
            <button 
              onClick={onNavigateToDashboard}
              className="flex items-center gap-1 hover:text-light-blue transition-colors"
            >
              <Home className="w-4 h-4" />
              <span>digifunzi</span>
            </button>
            <span className="text-gray-400">•</span>
            <div className="flex items-center gap-1">
              <div className="w-4 h-4 flex items-center justify-center">
                {classType === "crib" && <Heart className="w-4 h-4 text-light-blue" fill="currentColor" />}
                {classType === "clicker" && <Lightbulb className="w-4 h-4 text-light-blue" fill="currentColor" />}
                {classType === "quest" && <Shield className="w-4 h-4 text-navy" fill="currentColor" />}
              </div>
              <span>{config.className}</span>
            </div>
            {currentCourse && (
              <>
                <span className="text-gray-400">•</span>
                <span className="text-gray-900 font-medium">{currentCourse}</span>
              </>
            )}
          </div>
          
          <div className="flex items-center gap-2">
            <Badge className="bg-yellow text-white">
              Level 3
            </Badge>
            <Badge variant="outline" className="bg-light-blue/10 border-light-blue/30 text-light-blue">
              150 XP
            </Badge>
            <div className="flex items-center gap-1">
              <div className="w-6 h-6 bg-green-500 rounded-full flex items-center justify-center text-white font-medium text-xs">
                A
              </div>
              <div className="w-6 h-6 bg-yellow rounded-full flex items-center justify-center text-white font-medium text-xs">
                ♠
              </div>
              <div className="w-6 h-6 bg-orange-500 rounded-full flex items-center justify-center text-white font-medium text-xs">
                S
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}