import { useState } from "react";
import { Link } from "react-router-dom";
import { Card, CardContent } from "./ui/card";
import { Badge } from "./ui/badge";
import { Button } from "./ui/button";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription } from "./ui/dialog";
import { Play, Lock, Heart, Lightbulb, Shield, Video } from "lucide-react";
import { VideoPlayer } from "./VideoPlayer";
import { PaymentModal } from "./PaymentModal";
import { ImageWithFallback } from "./figma/ImageWithFallback";

interface Course {
  id: string;
  title: string;
  sessions: number;
  currentSession: number;
  isPaid: boolean;
  isCompleted: boolean;
  price?: number;
}

interface ClassCardProps {
  classType: "crib" | "clicker" | "quest";
  title: string;
  description: string;
  courses: Course[];
  onViewCourse: (courseId: string) => void;
  onPayForCourse: (courseId: string) => void;
}

export function ClassCard({ 
  classType, 
  title, 
  description, 
  courses, 
  onViewCourse, 
  onPayForCourse 
}: ClassCardProps) {
  const [selectedCourse, setSelectedCourse] = useState<Course | null>(null);
  const [isPaymentModalOpen, setIsPaymentModalOpen] = useState(false);
  const [showIntroVideo, setShowIntroVideo] = useState<string | null>(null);

  const getClassConfig = () => {
    switch (classType) {
      case "crib":
        return {
          gradient: "gradient-crib-card",
          icon: <Heart className="w-8 h-8 text-white" fill="currentColor" />,
          badgeColor: "bg-navy/20 text-navy border-navy/30",
          buttonColor: "bg-light-blue hover:bg-light-blue/90 text-white",
          ageRange: "Ages 5-7 years",
          courseCount: `${courses.length} Adventures`,
          courseImages: [
            "https://images.unsplash.com/photo-1565229284535-2cbbe3049123?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxjb2RpbmclMjBibG9ja3MlMjBjb2xvcmZ1bCUyMGtpZHMlMjBwcm9ncmFtbWluZ3xlbnwxfHx8fDE3NTg4ODE0MzF8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
            "https://images.unsplash.com/photo-1603354350317-6f7aaa5911c5?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxjb2RpbmclMjBlZHVjYXRpb24lMjBraWRzfGVufDF8fHx8MTc1ODg4MTg1Mnww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
            "https://images.unsplash.com/photo-1674049406486-4b1f6e1845fd?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxwcm9ncmFtbWluZyUyMGNoaWxkcmVuJTIwdGVjaG5vbG9neXxlbnwxfHx8fDE3NTg4ODE4NTV8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral"
          ]
        };
      case "clicker":
        return {
          gradient: "gradient-clicker-card",
          icon: <Lightbulb className="w-8 h-8 text-yellow" fill="currentColor" />,
          badgeColor: "bg-yellow/20 text-navy border-yellow/30",
          buttonColor: "bg-yellow hover:bg-yellow/90 text-navy",
          ageRange: "Ages 8-12 years",
          courseCount: `${courses.length} Challenges`,
          courseImages: [
            "https://images.unsplash.com/photo-1717588282722-ab1beb899c26?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxnYW1lJTIwZGV2ZWxvcG1lbnQlMjBraWRzJTIwdGVjaG5vbG9neXxlbnwxfHx8fDE3NTg4ODE0MzR8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
            "https://images.unsplash.com/photo-1568585262983-9b54814595a9?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHxyb2JvdGljcyUyMGtpZHMlMjBsZWFybmluZ3xlbnwxfHx8fDE3NTg4ODE4NTd8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
            "https://images.unsplash.com/photo-1603354350317-6f7aaa5911c5?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxjb2RpbmclMjBlZHVjYXRpb24lMjBraWRzfGVufDF8fHx8MTc1ODg4MTg1Mnww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral"
          ]
        };
      case "quest":
        return {
          gradient: "gradient-quest-card",
          icon: <Shield className="w-8 h-8 text-white" fill="currentColor" />,
          badgeColor: "bg-navy/20 text-navy border-navy/30",
          buttonColor: "bg-navy hover:bg-navy/90 text-white",
          ageRange: "Ages 13+ years",
          courseCount: `${courses.length} Courses`,
          courseImages: [
            "https://images.unsplash.com/photo-1573496528681-9b0f4fb0c660?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxweXRob24lMjBwcm9ncmFtbWluZyUyMGFkdmFuY2VkJTIwY29kaW5nfGVufDF8fHx8MTc1ODg4MTQzN3ww&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
            "https://images.unsplash.com/photo-1674049406486-4b1f6e1845fd?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxwcm9ncmFtbWluZyUyMGNoaWxkcmVuJTIwdGVjaG5vbG9neXxlbnwxfHx8fDE3NTg4ODE4NTV8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral",
            "https://images.unsplash.com/photo-1568585262983-9b54814595a9?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHxyb2JvdGljcyUyMGtpZHMlMjBsZWFybmluZ3xlbnwxfHx8fDE3NTg4ODE4NTd8MA&ixlib=rb-4.1.0&q=80&w=1080&utm_source=figma&utm_medium=referral"
          ]
        };
    }
  };

  const config = getClassConfig();

  const handleUnlockCourse = (course: Course) => {
    setSelectedCourse(course);
    setIsPaymentModalOpen(true);
  };

  const handlePaymentSuccess = () => {
    if (selectedCourse) {
      onPayForCourse(selectedCourse.id);
    }
  };

  return (
    <>
      <Card className="overflow-hidden
                      border-0
                      shadow-lg">
        {/* Header section - optimized for full width */}
        <div className={`${config.gradient} relative`}>
          <div className="flex
                         flex-col
                         lg:flex-row
                         lg:items-center
                         justify-between
                         p-6
                         text-white
                         gap-4">
            <div className="flex-1
                           max-w-2xl">
              <div className="flex
                             items-center
                             gap-3
                             mb-3">
                <div className="flex-shrink-0">
                  {config.icon}
                </div>
                <h3 className="text-2xl
                              font-semibold">{title}</h3>
              </div>
              <p className="text-white/90
                          leading-relaxed">{description}</p>
            </div>
            
            {/* Age and course info - moved to right side */}
            <div className="flex
                           lg:flex-col
                           gap-3
                           lg:items-end">
              <Badge className={`${config.badgeColor}
                                bg-white/20
                                text-white
                                border-white/30`}>
                {config.ageRange}
              </Badge>
              <Badge variant="outline" 
                     className="border-white/30
                               bg-white/10
                               text-white">
                {config.courseCount}
              </Badge>
            </div>
          </div>
        </div>

        <CardContent className="p-6">
          {/* All Courses List - Optimized for horizontal layout */}
          <div>
            <h4 className="font-medium
                          text-gray-900
                          mb-6">All Courses</h4>
            
            <div className="grid
                           grid-cols-1
                           lg:grid-cols-2
                           xl:grid-cols-3
                           gap-4">
              {courses.map((course, index) => (
                <div key={course.id} 
                     className="flex
                               items-center
                               gap-4
                               p-4
                               bg-white
                               rounded-lg
                               border
                               hover:shadow-md
                               transition-shadow">
                  <div className="w-16
                                 h-16
                                 rounded-lg
                                 overflow-hidden
                                 flex-shrink-0">
                    <ImageWithFallback
                      src={config.courseImages[index % config.courseImages.length]}
                      alt={course.title}
                      className="w-full
                                h-full
                                object-cover"
                    />
                  </div>
                  
                  <div className="flex-1
                                 min-w-0">
                    <h5 className="font-medium
                                  text-gray-900
                                  mb-1
                                  truncate">{course.title}</h5>
                    <p className="text-sm
                                 text-gray-600">Programming Course</p>
                    
                    <div className="flex
                                   items-center
                                   gap-2
                                   mt-3">
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => setShowIntroVideo(course.id)}
                        className="text-gray-600
                                  border-gray-300
                                  h-8
                                  px-2"
                      >
                        <Video className="w-3
                                         h-3" />
                      </Button>
                      
                      {course.isPaid ? (
                        <Link
                          to={`/course/${classType}/${course.id}`}
                          className={`${config.buttonColor}
                                     h-8
                                     px-3
                                     inline-flex
                                     items-center
                                     justify-center
                                     rounded-md
                                     text-sm
                                     transition-colors
                                     focus-visible:outline-none
                                     focus-visible:ring-2
                                     focus-visible:ring-ring
                                     focus-visible:ring-offset-2
                                     disabled:opacity-50
                                     disabled:pointer-events-none
                                     ring-offset-background`}
                        >
                          <Play className="w-3
                                          h-3
                                          mr-1" />
                          Start
                        </Link>
                      ) : (
                        <Button
                          size="sm"
                          onClick={() => handleUnlockCourse(course)}
                          className={classType === "crib" 
                            ? "gradient-crib-button text-white h-8 px-3 border-0 shadow-md"
                            : "bg-yellow hover:bg-yellow/90 text-navy h-8 px-3 border-0"
                          }
                        >
                          <Lock className="w-3
                                          h-3
                                          mr-1" />
                          ${course.price}
                        </Button>
                      )}
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Course Introduction Video Dialog */}
      <Dialog open={!!showIntroVideo} onOpenChange={() => setShowIntroVideo(null)}>
        <DialogContent className="max-w-4xl">
          <DialogHeader>
            <DialogTitle>
              {courses.find(c => c.id === showIntroVideo)?.title} - Course Introduction
            </DialogTitle>
            <DialogDescription>
              Watch this introduction video to learn about the course content and objectives.
            </DialogDescription>
          </DialogHeader>
          <div className="mt-4">
            <VideoPlayer 
              title={`${courses.find(c => c.id === showIntroVideo)?.title} - Introduction`}
              className="w-full
                        rounded-lg
                        overflow-hidden"
            />
          </div>
        </DialogContent>
      </Dialog>

      {/* Payment Modal */}
      {selectedCourse && (
        <PaymentModal
          isOpen={isPaymentModalOpen}
          onClose={() => {
            setIsPaymentModalOpen(false);
            setSelectedCourse(null);
          }}
          courseTitle={selectedCourse.title}
          price={selectedCourse.price || 0}
          onPaymentSuccess={handlePaymentSuccess}
          classType={classType}
        />
      )}
    </>
  );
}