import React from "react";
import { Card, CardContent, CardHeader, CardTitle } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { 
  Receipt, 
  DollarSign, 
  CreditCard, 
  Calendar, 
  Download,
  Clock,
  CheckCircle,
  AlertCircle,
  FileText,
  Smartphone
} from "lucide-react";
import { PaymentModal } from "./PaymentModal";
import { toast } from "sonner@2.0.3";

export class BillingsView extends React.Component {
  constructor(props) {
    super(props);
    
    this.state = {
      showPaymentModal: false,
      selectedInvoice: null
    };

    // Mock data for invoices
    this.invoices = [
      {
        id: "1",
        courseTitle: "Python Programming Mastery",
        amount: 150,
        dueDate: "2025-01-15",
        issueDate: "2024-12-15",
        status: "pending",
        description: "Full course access - 16 sessions",
        classType: "quest",
        invoiceNumber: "INV-2024-001"
      },
      {
        id: "2", 
        courseTitle: "Game Builder Academy",
        amount: 120,
        dueDate: "2024-12-20",
        issueDate: "2024-11-20",
        status: "paid",
        description: "Full course access - 15 sessions",
        classType: "clicker",
        invoiceNumber: "INV-2024-002"
      },
      {
        id: "3",
        courseTitle: "Magic Code Blocks Adventure",
        amount: 80,
        dueDate: "2024-12-10",
        issueDate: "2024-11-10", 
        status: "overdue",
        description: "Full course access - 12 sessions",
        classType: "crib",
        invoiceNumber: "INV-2024-003"
      },
      {
        id: "4",
        courseTitle: "Robot Programming Adventures", 
        amount: 100,
        dueDate: "2025-01-05",
        issueDate: "2024-12-05",
        status: "pending",
        description: "Full course access - 10 sessions",
        classType: "clicker",
        invoiceNumber: "INV-2024-004"
      },
      {
        id: "5",
        courseTitle: "Rainbow Digital Art Studio",
        amount: 75,
        dueDate: "2024-11-25", 
        issueDate: "2024-10-25",
        status: "paid",
        description: "Full course access - 10 sessions",
        classType: "crib",
        invoiceNumber: "INV-2024-005"
      }
    ];

    // Mock data for payments
    this.payments = [
      {
        id: "1",
        amount: 120,
        date: "2024-12-18",
        method: "mpesa",
        courseTitle: "Game Builder Academy",
        transactionId: "MP241218001",
        status: "completed"
      },
      {
        id: "2",
        amount: 75,
        date: "2024-11-23",
        method: "mpesa", 
        courseTitle: "Rainbow Digital Art Studio",
        transactionId: "MP241123001",
        status: "completed"
      },
      {
        id: "3",
        amount: 50,
        date: "2024-12-01",
        method: "card",
        courseTitle: "Technical Mentor Session",
        transactionId: "CD241201001", 
        status: "completed"
      }
    ];

    // Bind methods
    this.handlePayInvoice = this.handlePayInvoice.bind(this);
    this.handlePaymentSuccess = this.handlePaymentSuccess.bind(this);
    this.downloadInvoice = this.downloadInvoice.bind(this);
    this.getStatusBadge = this.getStatusBadge.bind(this);
    this.getPaymentMethodIcon = this.getPaymentMethodIcon.bind(this);
    this.getClassTypeColor = this.getClassTypeColor.bind(this);
  }

  getStatusBadge(status) {
    switch (status) {
      case "paid":
        return <Badge className="bg-green-100 text-green-700">✓ Paid</Badge>;
      case "pending":
        return <Badge className="bg-yellow-100 text-yellow-700">⏳ Pending</Badge>;
      case "overdue":
        return <Badge className="bg-red-100 text-red-700">⚠️ Overdue</Badge>;
    }
  }

  getPaymentMethodIcon(method) {
    switch (method) {
      case "mpesa":
        return <Smartphone size={16} className="text-green-600" />;
      case "card":
        return <CreditCard size={16} className="text-blue-600" />;
      case "bank":
        return <DollarSign size={16} className="text-purple-600" />;
    }
  }

  getClassTypeColor(classType) {
    switch (classType) {
      case "crib":
        return "border-l-yellow";
      case "clicker": 
        return "border-l-light-blue";
      case "quest":
        return "border-l-navy";
    }
  }

  handlePayInvoice(invoice) {
    this.setState({
      selectedInvoice: invoice,
      showPaymentModal: true
    });
  }

  handlePaymentSuccess() {
    const { selectedInvoice } = this.state;
    if (selectedInvoice) {
      toast.success(`Payment successful for ${selectedInvoice.courseTitle}!`);
      this.setState({
        showPaymentModal: false,
        selectedInvoice: null
      });
    }
  }

  downloadInvoice(invoiceId, courseTitle) {
    toast.success(`Downloading invoice for ${courseTitle}...`);
  }

  render() {
    const { showPaymentModal, selectedInvoice } = this.state;

    // Calculate totals
    const totalOwed = this.invoices
      .filter(inv => inv.status === "pending" || inv.status === "overdue")
      .reduce((sum, inv) => sum + inv.amount, 0);
    
    const totalPaid = this.payments
      .filter(payment => payment.status === "completed")
      .reduce((sum, payment) => sum + payment.amount, 0);

    const overdueAmount = this.invoices
      .filter(inv => inv.status === "overdue")
      .reduce((sum, inv) => sum + inv.amount, 0);

    return (
      <div className="p-6 max-w-7xl mx-auto">
        {/* Header */}
        <div className="mb-8">
          <h1 className="text-3xl mb-2">💳 Billing & Payments</h1>
          <p className="text-gray-600">Manage your course payments, invoices, and billing history</p>
        </div>

        {/* Summary Cards */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600 mb-1">Total Amount Owed</p>
                  <p className="text-2xl text-red-600">${totalOwed}</p>
                </div>
                <div className="p-3 bg-red-100 rounded-full">
                  <AlertCircle className="text-red-600" size={24} />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600 mb-1">Total Paid</p>
                  <p className="text-2xl text-green-600">${totalPaid}</p>
                </div>
                <div className="p-3 bg-green-100 rounded-full">
                  <CheckCircle className="text-green-600" size={24} />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600 mb-1">Overdue Amount</p>
                  <p className="text-2xl text-orange-600">${overdueAmount}</p>
                </div>
                <div className="p-3 bg-orange-100 rounded-full">
                  <Clock className="text-orange-600" size={24} />
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Main Content */}
        <Tabs defaultValue="invoices" className="space-y-6">
          <TabsList className="grid w-full grid-cols-2">
            <TabsTrigger value="invoices" className="flex items-center gap-2">
              <Receipt size={16} />
              Invoices
            </TabsTrigger>
            <TabsTrigger value="payments" className="flex items-center gap-2">
              <CreditCard size={16} />
              Payment History
            </TabsTrigger>
          </TabsList>

          {/* Invoices Tab */}
          <TabsContent value="invoices">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Receipt size={20} />
                  Course Invoices
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {this.invoices.map((invoice) => (
                    <Card key={invoice.id} className={`border-l-4 ${this.getClassTypeColor(invoice.classType)}`}>
                      <CardContent className="p-4">
                        <div className="flex items-center justify-between">
                          <div className="flex-1">
                            <div className="flex items-center gap-3 mb-2">
                              <h3 className="font-medium">{invoice.courseTitle}</h3>
                              {this.getStatusBadge(invoice.status)}
                            </div>
                            
                            <div className="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm text-gray-600">
                              <div className="flex items-center gap-2">
                                <FileText size={14} />
                                <span>Invoice #{invoice.invoiceNumber}</span>
                              </div>
                              <div className="flex items-center gap-2">
                                <Calendar size={14} />
                                <span>Due: {new Date(invoice.dueDate).toLocaleDateString()}</span>
                              </div>
                              <div className="flex items-center gap-2">
                                <DollarSign size={14} />
                                <span className="font-medium">${invoice.amount}</span>
                              </div>
                            </div>
                            
                            <p className="text-sm text-gray-500 mt-2">{invoice.description}</p>
                          </div>
                          
                          <div className="flex items-center gap-2 ml-4">
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() => this.downloadInvoice(invoice.id, invoice.courseTitle)}
                            >
                              <Download size={14} className="mr-1" />
                              Download
                            </Button>
                            
                            {invoice.status !== "paid" && (
                              <Button
                                size="sm"
                                onClick={() => this.handlePayInvoice(invoice)}
                                className={`${
                                  invoice.status === "overdue" 
                                    ? "bg-red-600 hover:bg-red-700" 
                                    : "bg-light-blue hover:bg-light-blue/90"
                                }`}
                              >
                                <CreditCard size={14} className="mr-1" />
                                Pay Now
                              </Button>
                            )}
                          </div>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Payments Tab */}
          <TabsContent value="payments">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <CreditCard size={20} />
                  Payment History
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {this.payments.map((payment) => (
                    <Card key={payment.id} className="border">
                      <CardContent className="p-4">
                        <div className="flex items-center justify-between">
                          <div className="flex-1">
                            <div className="flex items-center gap-3 mb-2">
                              <h3 className="font-medium">{payment.courseTitle}</h3>
                              <Badge className="bg-green-100 text-green-700">
                                <CheckCircle size={12} className="mr-1" />
                                Completed
                              </Badge>
                            </div>
                            
                            <div className="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm text-gray-600">
                              <div className="flex items-center gap-2">
                                {this.getPaymentMethodIcon(payment.method)}
                                <span className="capitalize">
                                  {payment.method === "mpesa" ? "M-Pesa" : payment.method}
                                </span>
                              </div>
                              <div className="flex items-center gap-2">
                                <Calendar size={14} />
                                <span>{new Date(payment.date).toLocaleDateString()}</span>
                              </div>
                              <div className="flex items-center gap-2">
                                <span className="text-xs text-gray-500">
                                  ID: {payment.transactionId}
                                </span>
                              </div>
                            </div>
                          </div>
                          
                          <div className="text-right ml-4">
                            <p className="text-lg font-medium text-green-600">
                              ${payment.amount}
                            </p>
                            <Button variant="outline" size="sm" className="mt-2">
                              <Download size={14} className="mr-1" />
                              Receipt
                            </Button>
                          </div>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>

        {/* Payment Modal */}
        {showPaymentModal && selectedInvoice && (
          <PaymentModal
            isOpen={showPaymentModal}
            onClose={() => this.setState({ showPaymentModal: false })}
            onPaymentSuccess={this.handlePaymentSuccess}
            courseTitle={selectedInvoice.courseTitle}
            price={selectedInvoice.amount}
          />
        )}
      </div>
    );
  }
}
