# DigiFunzi Learner Dashboard - TypeScript Edition

A comprehensive learner management system built with React, TypeScript, and Tailwind CSS, featuring age-appropriate interfaces for three distinct learning groups.

## 🎯 Overview

DigiFunzi is an educational platform designed to provide age-appropriate coding education for three distinct groups:

- **🌈 Crib (5-7 years)**: Magical, colorful, story-based learning experience
- **⚡ Clicker (8-12 years)**: Game-focused, tech-oriented, achievement-based interface  
- **💻 Quest (13+ years)**: Professional, career-focused, industry-standard approach

## 🏗️ Architecture

### TypeScript Structure

```
├── App.tsx                    # Main application entry point
├── types/
│   └── index.ts              # Comprehensive type definitions
├── utils/
│   └── helpers.ts            # Utility functions with full type safety
├── components/
│   ├── *.tsx                 # All components with TypeScript
│   └── ui/                   # Reusable UI components (shadcn/ui)
└── styles/
    └── globals.css           # Tailwind v4 with custom DigiFunzi theme
```

### Key Type Definitions

```typescript
// Age-based class categories
export type ClassType = "crib" | "clicker" | "quest";

// Core interfaces
export interface CurrentLesson {
  courseId: string;
  courseTitle: string;
  currentSession: number;
  totalSessions: number;
  classType: ClassType;
}

export interface SelectedMentor {
  id: string;
  name: string;
  hourlyRate: number;
  expertise: string[];
  rating?: number;
}
```

## 🎨 Design System

### Brand Colors
```css
--navy: #25476a;      /* Primary - Professional/Quest */
--yellow: #feb139;    /* Primary - Playful/Crib */
--light-blue: #38aae1; /* Primary - Engaging/Clicker */
```

### Age-Appropriate Theming
- **Crib**: Yellow to Light Blue gradients, playful emojis, magical terminology
- **Clicker**: Light Blue to Navy gradients, game-focused UI, achievement badges
- **Quest**: Navy to Light Blue gradients, professional styling, career terminology

## 🔧 Core Features

### 1. **Age-Appropriate Interfaces**
```typescript
export const AGE_GROUP_CONFIGS: Record<ClassType, AgeGroupConfig> = {
  crib: {
    theme: { primaryColor: "#feb139", gradientFrom: "#feb139", gradientTo: "#38aae1" },
    terminology: { assignments: "Fun Projects", sessions: "Learning Adventures" },
    contentStyle: "playful"
  },
  // ... other configurations
};
```

### 2. **Progressive Session Unlocking**
```typescript
export function isSessionUnlocked(
  sessionNumber: number, 
  currentSession: number, 
  sessionProgress: SessionProgress[]
): boolean {
  // Logic ensures users complete sessions sequentially
}
```

### 3. **M-Pesa Payment Integration**
```typescript
export function validateMpesaNumber(phoneNumber: string): boolean {
  const kenyaPhoneRegex = /^(\+254|0)[17]\d{8}$/;
  return kenyaPhoneRegex.test(phoneNumber.replace(/\s/g, ''));
}
```

### 4. **Mentor Booking System**
```typescript
export interface MentorBooking {
  id: string;
  mentorId: string;
  timeSlot: TimeSlot;
  sessionType: "homework_help" | "project_review" | "concept_explanation";
  status: "pending" | "confirmed" | "completed" | "cancelled";
}
```

## 📱 Component Architecture

### Main Components

1. **App.tsx** - Central state management and routing logic
2. **Sidebar.tsx** - Navigation with type-safe menu items
3. **DashboardView.tsx** - Age-appropriate dashboard with analytics
4. **ClassesView.tsx** - Course catalog with payment integration
5. **LessonView.tsx** - Interactive learning interface with video player
6. **MentorBooking.tsx** - Mentor scheduling and payment system

### UI Components (shadcn/ui)
- Fully typed components with consistent styling
- Custom theme integration with DigiFunzi brand colors
- Responsive design for all screen sizes

## 🎓 Learning Management

### Session Structure
```typescript
export interface LessonContent {
  introduction?: VideoContent;
  activities: VideoContent[];
  assignment?: VideoContent;
  resources?: Resource[];
}
```

### Progress Tracking
```typescript
export interface SessionProgress {
  sessionNumber: number;
  isCompleted: boolean;
  completionDate?: string;
  timeSpent?: number;
  score?: number;
}
```

## 💰 Payment & Billing

### M-Pesa Integration
- Phone number validation for Kenyan mobile formats
- Secure payment processing for course unlocks
- Mentor session payment handling

### Billing Records
```typescript
export interface BillingRecord {
  id: string;
  date: string;
  description: string;
  amount: number;
  status: "paid" | "pending" | "failed";
  paymentMethod: string;
}
```

## 🛠️ Development Setup

### Prerequisites
- Node.js 18+
- TypeScript 4.9+
- React 18+

### Installation
```bash
npm install
npm run dev
```

### Type Checking
```bash
npm run type-check
```

## 🎯 Age-Appropriate Features

### Crib (5-7 years)
- 🌟 Magical terminology ("Magic Code Blocks Adventure")
- 🎨 Bright, colorful interfaces with yellow/light-blue gradients
- 🎭 Story-based learning with character avatars
- 🌈 Simple, large UI elements

### Clicker (8-12 years)  
- ⚡ Gaming terminology ("Game Builder Academy", "Level up!")
- 🎮 Achievement badges and progress bars
- 🚀 Tech-focused language and interface elements
- 🎯 Interactive elements with hover effects

### Quest (13+ years)
- 💻 Professional terminology ("Full Stack Development Bootcamp")
- 📊 Industry-standard interface design
- 🎖️ Career-focused achievements and certifications
- 📈 Advanced analytics and progress tracking

## 🔒 Type Safety

### Utility Functions
```typescript
// Age validation with type safety
export function validateAgeGroup(birthDate: string, classType: ClassType): boolean {
  const age = calculateAge(birthDate);
  
  switch (classType) {
    case 'crib': return age >= 5 && age <= 7;
    case 'clicker': return age >= 8 && age <= 12;
    case 'quest': return age >= 13;
    default: return false;
  }
}
```

### Error Handling
```typescript
export interface ApiResponse<T> {
  success: boolean;
  data?: T;
  message?: string;
  error?: string;
}
```

## 🌟 Key Benefits

1. **Type Safety**: Comprehensive TypeScript coverage prevents runtime errors
2. **Age Appropriateness**: Tailored interfaces for different age groups
3. **Scalability**: Modular architecture supports easy feature additions
4. **User Experience**: Age-specific terminology and visual design
5. **Payment Integration**: Secure M-Pesa integration for Kenyan market
6. **Mentor System**: Built-in support for paid technical help
7. **Progress Tracking**: Comprehensive learning analytics

## 🚀 Deployment

The TypeScript version provides:
- Better IDE support and autocomplete
- Compile-time error detection
- Improved code maintainability
- Enhanced team collaboration
- Better documentation through types

## 📊 Performance

- **Lazy Loading**: Components load on demand
- **Type-Safe State Management**: Prevents state-related bugs
- **Optimized Rendering**: React patterns with TypeScript benefits
- **Bundle Size**: Tree-shaking with TypeScript support

---

**DigiFunzi** - Empowering the next generation of coders with age-appropriate, type-safe learning experiences! 🚀✨