import { useState, useEffect } from "react";
import { BrowserRouter, Routes, Route, useNavigate, useParams } from "react-router-dom";
import { Sidebar } from "./components/Sidebar";
import { DashboardView } from "./components/DashboardView"; 
import { ClassesView } from "./components/ClassesView";
import { LessonView } from "./components/LessonView";
import { MentorBooking } from "./components/MentorBooking";
import { BillingsView } from "./components/BillingsView";
import { ClassSelectionView } from "./components/ClassSelectionView";
import { ProjectsView } from "./components/ProjectsView";
import { toast } from "sonner@2.0.3";
import { Toaster } from "./components/ui/sonner";

// TypeScript Interfaces for DigiFunzi System
interface CurrentLesson {
  courseId: string;
  courseTitle: string;
  currentSession: number;
  totalSessions: number;
  classType: "crib" | "clicker" | "quest";
}

interface SelectedMentor {
  id: string;
  name: string;
  hourlyRate: number;
  expertise: string[];
  avatar?: string;
  rating?: number;
  experience?: string;
}

interface CourseData {
  title: string;
  currentSession: number;
  totalSessions: number;
}

type SectionType = "dashboard" | "classes" | "projects" | "billing" | "profile" | "settings" | "lesson" | "mentor-booking";

// Age-appropriate course data with TypeScript typing
const courseDataMap: Record<string, CourseData> = {
  // Crib Courses (Ages 5-7) - Magical, colorful, story-based
  "crib-1": { title: "Magic Code Blocks Adventure", currentSession: 3, totalSessions: 12 },
  "crib-2": { title: "Rainbow Digital Art Studio", currentSession: 1, totalSessions: 10 },
  "crib-3": { title: "Fun Animations for Beginners", currentSession: 1, totalSessions: 8 },
  
  // Clicker Courses (Ages 8-12) - Game-focused, tech-oriented, achievement-based
  "clicker-1": { title: "Game Builder Academy", currentSession: 7, totalSessions: 15 },
  "clicker-2": { title: "Website Wizard Course", currentSession: 1, totalSessions: 12 },
  "clicker-3": { title: "Robot Programming Adventures", currentSession: 1, totalSessions: 10 },
  
  // Quest Courses (Ages 13+) - Professional, career-focused, industry-standard
  "quest-1": { title: "Python Programming Mastery", currentSession: 5, totalSessions: 16 },
  "quest-2": { title: "Full Stack Development Bootcamp", currentSession: 1, totalSessions: 14 },
  "quest-3": { title: "Advanced AI & Machine Learning", currentSession: 1, totalSessions: 12 },
};

// Component to handle course route and redirect to lesson view
interface CourseRedirectProps {
  onViewCourse: (courseId: string, classType: "crib" | "clicker" | "quest") => void;
}

function CourseRedirect({ onViewCourse }: CourseRedirectProps): null {
  const { classType, courseId } = useParams<{ classType: "crib" | "clicker" | "quest"; courseId: string }>();
  const navigate = useNavigate();

  useEffect(() => {
    if (courseId && classType) {
      onViewCourse(courseId, classType);
      navigate('/');
    }
  }, [courseId, classType, onViewCourse, navigate]);

  return null;
}

function AppContent(): JSX.Element {
  const [selectedClass, setSelectedClass] = useState<"crib" | "clicker" | "quest" | null>(null);
  const [activeSection, setActiveSection] = useState<SectionType>("dashboard");
  const [currentLesson, setCurrentLesson] = useState<CurrentLesson | null>(null);
  const [selectedMentor, setSelectedMentor] = useState<SelectedMentor | null>(null);

  const handleViewCourse = (courseId: string, classType: "crib" | "clicker" | "quest"): void => {
    const course = courseDataMap[courseId];
    if (course) {
      setCurrentLesson({
        courseId,
        courseTitle: course.title,
        currentSession: course.currentSession,
        totalSessions: course.totalSessions,
        classType: classType
      });
      setActiveSection("lesson");
    }
  };

  const handlePayForCourse = (courseId: string): void => {
    toast.success(`Payment initiated for course ${courseId}. M-Pesa integration would handle the payment flow.`);
  };

  const handleSessionComplete = (): void => {
    if (currentLesson) {
      const newSession = Math.min(currentLesson.currentSession + 1, currentLesson.totalSessions);
      setCurrentLesson({
        ...currentLesson,
        currentSession: newSession
      });
      
      if (newSession <= currentLesson.totalSessions) {
        toast.success(`🎉 Session ${currentLesson.currentSession} completed! Moving to session ${newSession}.`);
      } else {
        toast.success("🏆 Congratulations! You've completed the entire course!");
      }
    }
  };

  const handleBackToClasses = (): void => {
    setCurrentLesson(null);
    setActiveSection("classes");
  };

  const handleNavigateToDashboard = (): void => {
    setCurrentLesson(null);
    setSelectedMentor(null);
    setActiveSection("dashboard");
  };

  const handleNavigateToClasses = (): void => {
    setActiveSection("classes");
  };

  const handleNavigateToSession = (sessionNumber: number): void => {
    if (currentLesson && 
        sessionNumber >= 1 && 
        sessionNumber <= currentLesson.totalSessions && 
        sessionNumber <= currentLesson.currentSession + 1) {
      setCurrentLesson({
        ...currentLesson,
        currentSession: sessionNumber
      });
    }
  };

  const handleSelectMentor = (mentor: SelectedMentor): void => {
    setSelectedMentor(mentor);
    setActiveSection("mentor-booking");
  };

  const handleBackFromBooking = (): void => {
    setSelectedMentor(null);
    if (currentLesson) {
      setActiveSection("lesson");
    } else {
      setActiveSection("classes");
    }
  };

  const handleSectionChange = (section: string): void => {
    setActiveSection(section as SectionType);
    // Clear lesson and mentor state when navigating to main sections
    if (!["lesson", "mentor-booking"].includes(section)) {
      setCurrentLesson(null);
      setSelectedMentor(null);
    }
  };

  const handleClassSelection = (classType: "crib" | "clicker" | "quest"): void => {
    setSelectedClass(classType);
    toast.success(`Welcome to ${classType === "crib" ? "Crib Adventures" : classType === "clicker" ? "Clicker Academy" : "Quest Mastery"}! 🎉`);
  };

  const handleChangeClass = (): void => {
    setSelectedClass(null);
    setActiveSection("dashboard");
    setCurrentLesson(null);
    setSelectedMentor(null);
  };

  // Show class selection if no class is selected yet
  if (!selectedClass) {
    return (
      <div className="min-h-screen bg-gray-50">
        <ClassSelectionView onSelectClass={handleClassSelection} />
        <Toaster />
      </div>
    );
  }

  // Mentor booking view
  if (selectedMentor && activeSection === "mentor-booking") {
    return (
      <div className="min-h-screen bg-gray-50">
        <MentorBooking
          mentor={selectedMentor}
          onBack={handleBackFromBooking}
          courseTitle={currentLesson?.courseTitle}
          classType={currentLesson?.classType}
          onNavigateToDashboard={handleNavigateToDashboard}
          onBackToClasses={handleBackToClasses}
        />
        <Toaster />
      </div>
    );
  }

  // Lesson view
  if (currentLesson && activeSection === "lesson") {
    return (
      <div className="min-h-screen bg-gray-50">
        <LessonView
          courseTitle={currentLesson.courseTitle}
          currentSession={currentLesson.currentSession}
          totalSessions={currentLesson.totalSessions}
          classType={currentLesson.classType}
          onBack={handleBackToClasses}
          onSessionComplete={handleSessionComplete}
          onNavigateToSession={handleNavigateToSession}
          onSelectMentor={handleSelectMentor}
          onNavigateToDashboard={handleNavigateToDashboard}
        />
        <Toaster />
      </div>
    );
  }

  // Main dashboard layout
  return (
    <>
      <Routes>
        <Route path="/course/:classType/:courseId" element={<CourseRedirect onViewCourse={handleViewCourse} />} />
        <Route path="*" element={null} />
      </Routes>
      <div className="flex h-screen bg-gray-50">
        <Sidebar 
          activeSection={activeSection} 
          onSectionChange={handleSectionChange}
          selectedClass={selectedClass}
        />
        
        <main className="flex-1 overflow-auto">
          {activeSection === "dashboard" && selectedClass && (
            <DashboardView 
              onNavigateToClasses={handleNavigateToClasses} 
              selectedClass={selectedClass}
              onChangeClass={handleChangeClass}
            />
          )}
          
          {activeSection === "classes" && selectedClass && (
            <ClassesView 
              onViewCourse={handleViewCourse}
              onPayForCourse={handlePayForCourse}
              selectedClass={selectedClass}
            />
        )}
        
        {activeSection === "projects" && selectedClass && (
          <ProjectsView selectedClass={selectedClass} />
        )}
        
        {activeSection === "billing" && (
          <BillingsView />
        )}
        
        {activeSection === "profile" && selectedClass && (
          <div className="p-6">
            <div className={`mb-6 p-6 rounded-lg text-white ${
              selectedClass === "crib" ? "gradient-crib-header" :
              selectedClass === "clicker" ? "gradient-clicker-header" :
              "gradient-quest-header"
            }`}>
              <h1 className="text-3xl mb-2">
                {selectedClass === "crib" ? "⭐ My Stars & Achievements" :
                 selectedClass === "clicker" ? "👤 My Gamer Profile" :
                 "👤 Professional Profile"}
              </h1>
              <p className="text-white/90">
                {selectedClass === "crib" ? "See all the amazing things you've learned and created!" :
                 selectedClass === "clicker" ? "Track your achievements and level up your skills!" :
                 "Monitor your learning journey and professional development"}
              </p>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
              {/* Profile Info */}
              <div className="lg:col-span-2 bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                <h2 className="text-xl mb-4">
                  {selectedClass === "crib" ? "🎨 About Me" :
                   selectedClass === "clicker" ? "🎮 Player Info" :
                   "Profile Information"}
                </h2>
                <div className="space-y-4">
                  <div className="flex items-center gap-4">
                    <div className={`w-16 h-16 rounded-full flex items-center justify-center ${
                      selectedClass === "crib" ? "gradient-crib-avatar" :
                      selectedClass === "clicker" ? "gradient-clicker-avatar" :
                      "gradient-quest-avatar"
                    }`}>
                      <span className={`text-xl font-bold ${
                        selectedClass === "crib" ? "text-navy" : "text-white"
                      }`}>S</span>
                    </div>
                    <div>
                      <h3 className="text-lg">Student Name</h3>
                      <p className="text-gray-600">
                        {selectedClass === "crib" ? "Explorer Level 2 • Crib Adventures" :
                         selectedClass === "clicker" ? "Builder Level 2 • Clicker Academy" :
                         "Developer Level 2 • Quest Mastery"}
                      </p>
                      <p className="text-sm text-gray-500">Joined DigiFunzi 3 months ago</p>
                    </div>
                  </div>
                </div>
              </div>

              {/* Achievement Stats */}
              <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                <h3 className="mb-4">
                  {selectedClass === "crib" ? "🌟 My Stars" :
                   selectedClass === "clicker" ? "🏆 Achievements" :
                   "Achievements"}
                </h3>
                <div className="space-y-3">
                  <div className="flex items-center justify-between">
                    <span className="text-sm">
                      {selectedClass === "crib" ? "Adventures Done" :
                       selectedClass === "clicker" ? "Missions Completed" :
                       "Courses Completed"}
                    </span>
                    <span className="text-light-blue">3</span>
                  </div>
                  <div className="flex items-center justify-between">
                    <span className="text-sm">
                      {selectedClass === "crib" ? "Lessons Learned" :
                       selectedClass === "clicker" ? "Levels Beaten" :
                       "Sessions Finished"}
                    </span>
                    <span className="text-light-blue">47</span>
                  </div>
                  <div className="flex items-center justify-between">
                    <span className="text-sm">
                      {selectedClass === "crib" ? "Things Created" :
                       selectedClass === "clicker" ? "Projects Built" :
                       "Projects Created"}
                    </span>
                    <span className="text-light-blue">12</span>
                  </div>
                  <div className="flex items-center justify-between">
                    <span className="text-sm">
                      {selectedClass === "crib" ? "Helper Visits" :
                       selectedClass === "clicker" ? "Mentor Sessions" :
                       "Mentor Sessions"}
                    </span>
                    <span className="text-light-blue">5</span>
                  </div>
                </div>
              </div>
            </div>
          </div>
        )}
        
        {activeSection === "settings" && selectedClass && (
          <div className="p-6">
            <div className={`mb-6 p-6 rounded-lg text-white ${
              selectedClass === "crib" ? "gradient-crib-header" :
              selectedClass === "clicker" ? "gradient-clicker-header" :
              "gradient-quest-header"
            }`}>
              <h1 className="text-3xl mb-2">
                {selectedClass === "crib" ? "🎵 Fun Settings" :
                 selectedClass === "clicker" ? "⚙️ Game Settings" :
                 "⚙️ Preferences"}
              </h1>
              <p className="text-white/90">
                {selectedClass === "crib" ? "Make your learning space just the way you like it!" :
                 selectedClass === "clicker" ? "Customize your tech academy experience" :
                 "Customize your DigiFunzi learning experience"}
              </p>
            </div>

            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <h2 className="text-xl mb-4">
                {selectedClass === "crib" ? "My Choices" :
                 selectedClass === "clicker" ? "Settings & Options" :
                 "Settings & Preferences"}
              </h2>
              <div className="space-y-6">
                <div>
                  <h3 className="mb-2">
                    {selectedClass === "crib" ? "🔔 Reminder Bell" :
                     selectedClass === "clicker" ? "🔔 Mission Alerts" :
                     "Learning Notifications"}
                  </h3>
                  <label className="flex items-center gap-2">
                    <input type="checkbox" className="rounded" defaultChecked />
                    <span className="text-sm">
                      {selectedClass === "crib" ? "Tell me when it's time to learn!" :
                       selectedClass === "clicker" ? "Alert me about upcoming missions" :
                       "Remind me about upcoming sessions"}
                    </span>
                  </label>
                </div>

                <div>
                  <h3 className="mb-2">
                    {selectedClass === "crib" ? "👨‍👩‍👧 Share with Family" :
                     selectedClass === "clicker" ? "📊 Progress Sharing" :
                     "Progress Sharing"}
                  </h3>
                  <label className="flex items-center gap-2">
                    <input type="checkbox" className="rounded" />
                    <span className="text-sm">
                      {selectedClass === "crib" ? "Show my stars to my family" :
                       selectedClass === "clicker" ? "Share achievements with parents/guardians" :
                       "Share achievements with parents/guardians"}
                    </span>
                  </label>
                </div>

                <div>
                  <h3 className="mb-2">
                    {selectedClass === "crib" ? "🎂 My Age Group" :
                     selectedClass === "clicker" ? "📚 Learning Level" :
                     "Age Group"}
                  </h3>
                  <select className="w-full p-2 border border-gray-300 rounded-lg">
                    <option value="quest">Quest (13+ years)</option>
                    <option value="clicker">Clicker (8-12 years)</option>
                    <option value="crib">Crib (5-7 years)</option>
                  </select>
                </div>
              </div>
            </div>
          </div>
        )}
      </main>
      
      <Toaster />
    </div>
    </>
  );
}

export default function App(): JSX.Element {
  return (
    <BrowserRouter>
      <AppContent />
    </BrowserRouter>
  );
}